/** @file pd_alg_nb_sub_a_test.cpp  Unit Tests */


/* Pushdown plugin for FAU Discrete Event Systems Library (libfaudes)

   Copyright (C) 2013  Stefan Jacobi, Sven Schneider, Anne-Kathrin Hess

*/
#include "pd_alg_nb_sub_a_test.h"

namespace faudes {

/* *****************
 * TestFilterMixedGrammarSymbols
 * *****************/
void TestFilterMixedGrammarSymbols(){
  
  std::string name = "Filter Mixed Grammar Symbols";
  TestStart(name);
  
  Grammar gr1 = TestGrammar1();
  
  GrammarSymbolVector word;
  std::set<Terminal>::const_iterator tit;
  std::vector<TerminalPtr> tpv;
  std::vector<TerminalPtr>::iterator tpvit;
  std::set<Nonterminal> rSet;
  
  //build word to filter
  //contains one nonterminal and all terminals
  Terminal* t;
  Nonterminal* nt = new Nonterminal(*gr1.Nonterminals().begin());
  GrammarSymbolPtr ntPtr(nt);
  word.push_back(ntPtr);

  for(tit = gr1.Terminals().begin(); tit != gr1.Terminals().end(); tit++){
    //create new terminal
    t = new Terminal(*tit);
    TerminalPtr tPtr(t);
    //push terminal into word
    word.push_back(tPtr);
  }
  
  rSet = Filter(gr1.Nonterminals(),word);
  
  try{
    //resulting set size has to be one 
    if(rSet.size() != 1){
      std::stringstream errstr;
      errstr << "size of result set is " << rSet.size() << ", but 1 was expected" << std::endl;
      throw Exception(name, errstr.str(), 1003);
    }
    //resulting set has to contain *gr1.Nonterminals().begin()
    
    if(*rSet.begin() != *gr1.Nonterminals().begin()){
      std::stringstream errstr;
      errstr << "result set did not contain the expected nonterminal " << gr1.Nonterminals().begin()->Str() << std::endl;
      throw Exception(name, errstr.str(), 1003);
    }
  }
   catch(Exception e){
      std::cout << (*rSet.begin()).Str() << std::endl;
      std::cout << (*gr1.Nonterminals().begin()).Str() << std::endl;
  }
  
  TestEnd(name);
}

/* *****************
 * TestFilterNothing
 * *****************/
void TestFilterNothing(){
  std::string name = "Filter Nothing";
  TestStart(name);
  
  std::set<Nonterminal> nt;
  GrammarSymbolVector gs;
  
  std::set<Nonterminal> rSet = Filter(nt,gs);
  
  try{
    //resulting set size has to be zero 
    if(rSet.size() != 0){
      std::stringstream errstr;
      errstr << "size of result set is " << rSet.size() << ", but 0 was expected" << std::endl;
      throw Exception(name, errstr.str(), 1003);
    }
  }
   catch(Exception e){
  }
  
  TestEnd(name);
}

/* *****************
 * TestRnpp1FindSymbolsEmptySet
 * *****************/
void TestRnpp1FindSymbolsEmptySet(){
  std::string name = "Rnpp1 Find Symbols Empty Set";
  TestStart(name);
  
  Grammar gr = TestGrammar1();
  
  //eliminable nonterminal which should be found by Rnpp1
  std::vector<Idx> v;
  v.push_back(3);
  Nonterminal nt(5,v,1);
  //the grammar should contain this nonterminal!
  if(gr.Nonterminals().find(nt) == gr.Nonterminals().end()){
    std::cout << "Warning, test parameters seem to be wrong. The test grammar did not contain Nonterminal (5,[3],1)." << std::endl;
    return;
  }
  
  //empty set of already eliminated nonterminals
  std::set<Nonterminal> ntSet;
  
  std::set<Nonterminal> rSet = Rnpp1(gr,ntSet);
  
  try{
    //resulting set size has to be one 
    if(rSet.size() != 1){
      std::stringstream errstr;
      errstr << "size of result set is " << rSet.size() << ", but 1 was expected" << std::endl;
      throw Exception(name, errstr.str(), 1003);
    }
    //resulting set size has to contain nonterminal (3,2)
    if(*rSet.begin() != nt){
      std::stringstream errstr;
      errstr << "result set did not contain the expected nonterminal " << nt.Str() << std::endl;
      throw Exception(name, errstr.str(), 1003);
    }
  }
   catch(Exception e){
  }
  
  TestEnd(name);
}

/* *****************
 * TestRnpp1FindSymbolsNonemptySet
 * *****************/
void TestRnpp1FindSymbolsNonemptySet(){
  std::string name = "Rnpp1 Find Symbols Empty Set";
  TestStart(name);
  
  Grammar gr = TestGrammar1();
  
  //eliminable nonterminal which should be found
  std::vector<Idx> v;
  v.push_back(2);
  Nonterminal nt(1,v,2);
  //the grammar should contain this nonterminal!
  if(gr.Nonterminals().find(nt) == gr.Nonterminals().end()){
    std::cout << "Warning, test parameters seem to be wrong. The test grammar did not contain Nonterminal (1,[2],2)." << std::endl;
    std::cout << gr.Str() << std::endl;
    return;
  }
  
  //empty set of already eliminated nonterminals
  std::set<Nonterminal> ntSet;
  
  std::set<Nonterminal> rSet = Rnpp1(gr,Rnpp1(gr,ntSet));
  
  try{
    //resulting set size has to be one 
    if(rSet.size() != 2){
      std::stringstream errstr;
      errstr << "size of result set is " << rSet.size() << ", but 2 was expected" << std::endl;
      throw Exception(name, errstr.str(), 1003);
    }
    //resulting set size has to contain nonterminal (1,2,2)
    if(*rSet.begin() != nt){
      std::stringstream errstr;
      errstr << "result set did not contain the expected nonterminal " << nt.Str() << std::endl;
      throw Exception(name, errstr.str(), 1003);
    }
  }
   catch(Exception e){
  }
  
  TestEnd(name);
}

/* *****************
 * TestRnpplFindSymbolsEmptySet
 * *****************/
void TestRnpplFindSymbolsEmptySet(){
  std::string name = "Rnppl Find Symbols Empty Set";
  TestStart(name);
  
  Grammar gr = TestGrammar1();
  
  //all nonterminals should be eliminable except (2,3,3)
  std::set<Nonterminal> findThis = gr.Nonterminals();
  std::vector<Idx> v;
  v.push_back(3);
  findThis.erase(Nonterminal(2,v,3));
  
  //empty set of already eliminated nonterminals
  std::set<Nonterminal> ntSet;
  
  std::set<Nonterminal> rSet = Rnppl(gr,ntSet);
  
  try{
    //resulting set size has to be the size of findThis 
    if(rSet.size() != findThis.size()){
      std::stringstream errstr;
      errstr << "size of result set is " << rSet.size() << ", but " << findThis.size() << " was expected" << std::endl;
      throw Exception(name, errstr.str(), 1003);
    }
    //resulting set size has to contain all nonterminals in findThis
    std::set<Nonterminal>::const_iterator ntit, findit;
    for(ntit = findThis.begin(); ntit != findThis.end(); ntit++){
      //look for every nonterminal
      findit = rSet.find(*ntit);
      //throw error if it was not found
      if(findit == findThis.end()){
        std::stringstream errstr;
        errstr << "result set did not contain the expected nonterminal " << ntit->Str() << std::endl;
        throw Exception(name, errstr.str(), 1003);
      }
    }
  }
   catch(Exception e){
  }
  
  TestEnd(name);
}

/* *****************
 * TestRnpplFindSymbolsNonemptySet
 * *****************/
void TestRnpplFindSymbolsNonemptySet(){
  std::string name = "Rnppl Find Symbols Nonempty Set";
  TestStart(name);
  
  Grammar gr = TestGrammar1();
  
  //all nonterminals should be eliminable except (2,3,3)
  std::set<Nonterminal> findThis = gr.Nonterminals();
  std::vector<Idx> v;
  v.push_back(3);
  findThis.erase(Nonterminal(2,v,3));
  
  //set of already eliminated nonterminals, add any eliminable nonterminals
  std::set<Nonterminal> ntSet;
  v.clear(); v.push_back(2);
  ntSet.insert(Nonterminal(1,v,2));
  v.clear(); v.push_back(3);
  ntSet.insert(Nonterminal(4,v));
  
  std::set<Nonterminal> rSet = Rnppl(gr,ntSet);
  
  try{
    //resulting set size has to be the size of findThis
    if(rSet.size() != findThis.size()){
      std::stringstream errstr;
      errstr << "size of result set is " << rSet.size() << ", but " << findThis.size() << " was expected" << std::endl;
      throw Exception(name, errstr.str(), 1003);
    }
    //resulting set size has to contain all nonterminals in findThis
    std::set<Nonterminal>::const_iterator ntit, findit;
    for(ntit = findThis.begin(); ntit != findThis.end(); ntit++){
      //look for every nonterminal
      findit = rSet.find(*ntit);
      //throw error if it was not found
      if(findit == findThis.end()){
        std::stringstream errstr;
        errstr << "result set did not contain the expected nonterminal " << ntit->Str() << std::endl;
        throw Exception(name, errstr.str(), 1003);
      }
    }
  }
   catch(Exception e){
  }
  
  TestEnd(name);
}

/* *****************
 * TestRnpplFindSymbolsCompleteSet
 * *****************/
void TestRnpplFindSymbolsCompleteSet(){
  std::string name = "Rnppl Find Symbols Complete Set";
  TestStart(name);
  
  Grammar gr = TestGrammar1();
  
  //all nonterminals should be eliminable except (2,3,3)
  std::set<Nonterminal> findThis = gr.Nonterminals();
  std::vector<Idx> v;
  v.push_back(3);
  findThis.erase(Nonterminal(2,v,3));
  
  //set of already eliminated nonterminals, add any eliminable nonterminals
  std::set<Nonterminal> ntSet = findThis;
  
  std::set<Nonterminal> rSet = Rnppl(gr,ntSet);
  
  try{
    //resulting set size has to be the size of findThis
    if(rSet.size() != findThis.size()){
      std::stringstream errstr;
      errstr << "size of result set is " << rSet.size() << ", but " << findThis.size() << " was expected" << std::endl;
      throw Exception(name, errstr.str(), 1003);
    }
    //resulting set size has to contain all nonterminals in findThis
    std::set<Nonterminal>::const_iterator ntit, findit;
    for(ntit = findThis.begin(); ntit != findThis.end(); ntit++){
      //look for every nonterminal
      findit = rSet.find(*ntit);
      //throw error if it was not found
      if(findit == findThis.end()){
        std::stringstream errstr;
        errstr << "result set did not contain the expected nonterminal " << ntit->Str() << std::endl;
        throw Exception(name, errstr.str(), 1003);
      }
    }
  }
   catch(Exception e){
  }
  
  TestEnd(name);
}

/* *****************
 * TestRnppGrammar1
 * *****************/
void TestRnppGrammar1(){
  std::string name = "Rnpp Grammar 1";
  TestStart(name);
  
  Grammar gr = TestGrammar1();
  
  //all nonterminals should be eliminable except (2,3,3)
  std::set<Nonterminal> finalNtSet = gr.Nonterminals();
  std::vector<Idx> v;
  v.push_back(3);
  finalNtSet.erase(Nonterminal(2,v,3));
  //only one grammar production contains (2,3,3) and must be removed
  std::set<GrammarProduction> finalGpSet = gr.GrammarProductions();
  GrammarSymbolVector vg;
  Nonterminal* nt = new Nonterminal(2,v,3);
  GrammarSymbolPtr ntPtr(nt);
  vg.push_back(ntPtr);
  v.clear();  v.push_back(2);
  finalGpSet.erase(GrammarProduction(Nonterminal(1,v,2), vg));
  
  Grammar rGr = Rnpp(gr);
  
  try{
    //resulting size of nonterminals has to be the size of finalNtSet
    if(rGr.Nonterminals().size() != finalNtSet.size()){
      std::stringstream errstr;
      errstr << "size of nonterminal set is " << rGr.Nonterminals().size() << ", but " << finalNtSet.size() << " was expected" << std::endl;
      throw Exception(name, errstr.str(), 1003);
    }
    //resulting size of grammar productions has to be the size of finalGpSet
    if(rGr.GrammarProductions().size() != finalGpSet.size()){
      std::stringstream errstr;
      errstr << "size of grammar production set is " << rGr.GrammarProductions().size() << ", but " << finalGpSet.size() << " was expected" << std::endl;
      throw Exception(name, errstr.str(), 1003);
    }
    
    //resulting set size has to contain all nonterminals in finalNtSet
    std::set<Nonterminal>::const_iterator ntit, findntit;
    for(ntit = finalNtSet.begin(); ntit != finalNtSet.end(); ntit++){
      //look for every nonterminal
      findntit = rGr.Nonterminals().find(*ntit);
      //throw error if it was not found
      if(findntit == finalNtSet.end()){
        std::stringstream errstr;
        errstr << "nonterminal set did not contain the expected nonterminal " << ntit->Str() << std::endl;
        throw Exception(name, errstr.str(), 1003);
      }
    }
    
    //resulting set size has to contain all grammar productions in finalGpSet
    std::set<GrammarProduction>::const_iterator gpit, findgpit;
    for(gpit = finalGpSet.begin(); gpit != finalGpSet.end(); gpit++){
      //look for every nonterminal
      findgpit = rGr.GrammarProductions().find(*gpit);
      //throw error if it was not found
      if(findgpit == finalGpSet.end()){
        std::stringstream errstr;
        errstr << "grammar productions set did not contain the expected grammar production " << gpit->Str() << std::endl;
        throw Exception(name, errstr.str(), 1003);
      }
    }
  }
   catch(Exception e){
  }
  
  TestEnd(name);
}

/* *****************
 * TestRnppGrammar2
 * *****************/
void TestRnppGrammar2(){
  std::string name = "Rnpp Grammar 2";
  TestStart(name);
  
  Grammar gr = TestGrammar2();
  
  //all nonterminals should be eliminable except (2,3,3) and (3,2)
  std::vector<Idx> v2;
  v2.push_back(2);
  std::vector<Idx> v3;
  v3.push_back(3);
  std::set<Nonterminal> finalNtSet = gr.Nonterminals();
  finalNtSet.erase(Nonterminal(2,v3,3));
  finalNtSet.erase(Nonterminal(3,v2));
  //only one grammar production contains (2,3,3) and must be removed, no grammar
  //production contains (3,2)
  std::set<GrammarProduction> finalGpSet = gr.GrammarProductions();
  GrammarSymbolVector vg;
  Nonterminal* nt = new Nonterminal(2,v3,3);
  GrammarSymbolPtr ntPtr(nt);
  vg.push_back(ntPtr);
  finalGpSet.erase(GrammarProduction(Nonterminal(1,v2,2), vg));
  
  Grammar rGr = Rnpp(gr);
    
  try{
    //resulting size of nonterminals has to be the size of finalNtSet
    if(rGr.Nonterminals().size() != finalNtSet.size()){
      std::stringstream errstr;
      errstr << "size of nonterminal set is " << rGr.Nonterminals().size() << ", but " << finalNtSet.size() << " was expected" << std::endl;
      throw Exception(name, errstr.str(), 1003);
    }
    //resulting size of grammar productions has to be the size of finalGpSet
    if(rGr.GrammarProductions().size() != finalGpSet.size()){
      std::stringstream errstr;
      errstr << "size of grammar production set is " << rGr.GrammarProductions().size() << ", but " << finalGpSet.size() << " was expected" << std::endl;
      throw Exception(name, errstr.str(), 1003);
    }
    
    //resulting set size has to contain all nonterminals in finalNtSet
    std::set<Nonterminal>::const_iterator ntit, findntit;
    for(ntit = finalNtSet.begin(); ntit != finalNtSet.end(); ntit++){
      //look for every nonterminal
      findntit = rGr.Nonterminals().find(*ntit);
      //throw error if it was not found
      if(findntit == finalNtSet.end()){
        std::stringstream errstr;
        errstr << "nonterminal set did not contain the expected nonterminal " << ntit->Str() << std::endl;
        throw Exception(name, errstr.str(), 1003);
      }
    }
    
    //resulting set size has to contain all grammar productions in finalGpSet
    std::set<GrammarProduction>::const_iterator gpit, findgpit;
    for(gpit = finalGpSet.begin(); gpit != finalGpSet.end(); gpit++){
      //look for every nonterminal
      findgpit = rGr.GrammarProductions().find(*gpit);
      //throw error if it was not found
      if(findgpit == finalGpSet.end()){
        std::stringstream errstr;
        errstr << "grammar productions set did not contain the expected grammar production " << gpit->Str() << std::endl;
        throw Exception(name, errstr.str(), 1003);
      }
    }
  }
   catch(Exception e){
  }
  
  TestEnd(name);
}

/* *****************
 * TestRnppEmptyGrammar
 * *****************/
void TestRnppEmptyGrammar(){
  std::string name = "Rnpp Empty Grammar";
  TestStart(name);
  
  Grammar gr;
  
  Grammar rGr = Rnpp(gr);
  
  try{
    //resulting size of nonterminals has to be zero
    if(rGr.Nonterminals().size() != 0){
      std::stringstream errstr;
      errstr << "size of nonterminal set is " << rGr.Nonterminals().size() << ", but 0 was expected" << std::endl;
      throw Exception(name, errstr.str(), 1003);
    }
    //resulting size of grammar productions has to be zero
    if(rGr.GrammarProductions().size() != 0){
      std::stringstream errstr;
      errstr << "size of grammar production set is " << rGr.GrammarProductions().size() << ", but 0 was expected" << std::endl;
      throw Exception(name, errstr.str(), 1003);
    }
  }
   catch(Exception e){
  }
  
  TestEnd(name);
}

/* *****************
 * TestSp2LrTerminals
 * *****************/
void TestSp2LrTerminals(){
  std::string name = "Sp2Lr Terminals";
  TestStart(name);
  
  PushdownGenerator g = TestGenerator8();
  
  Grammar gr = Sp2Lr(g);
  
  try{
    //size of terminals must be the same as size of events
    if(g.Alphabet().Size() != gr.Terminals().size()){
      std::stringstream errstr;
      errstr << "size of terminal set is " << gr.Terminals().size() << ", but "<< g.Alphabet().Size() << " was expected" << std::endl;
      throw Exception(name, errstr.str(), 1003);
    }
    
    EventSet::Iterator eventit;
    //terminals must be all events
    for(eventit = g.AlphabetBegin(); eventit != g.AlphabetEnd(); eventit++){
      if(gr.Terminals().find(*eventit) == gr.Terminals().end()){
        std::stringstream errstr;
        errstr << "terminal " << g.EventName(*eventit) << " was not found in terminal set" << std::endl;
        throw Exception(name, errstr.str(), 1003);
      }
    }
  }
   catch(Exception e){
  }
  
  TestEnd(name);
}

/* *****************
 * TestSp2LrNonterminals
 * *****************/
void TestSp2LrNonterminals(){
  std::string name = "Sp2Lr Nonterminals";
  TestStart(name);
  
  PushdownGenerator g = TestGenerator8();
  
  Grammar gr = Sp2Lr(g);
  
  try{
    //number of all possible nonterminals
    uint expectedNumberNonterminals = g.States().Size() * (g.StackSymbols().Size() - 1) + g.States().Size() * g.States().Size() * (g.StackSymbols().Size() - 1);
    //if the size of nonterminals matches the expected size, none can be missing
    if(expectedNumberNonterminals != gr.Nonterminals().size()){
      std::stringstream errstr;
      errstr << "size of nonterminal set is " << gr.Nonterminals().size() << ", but "<< expectedNumberNonterminals << " was expected" << std::endl;
      throw Exception(name, errstr.str(), 1003);
    }
    
    //test for the correct starting nonterminal
    if(gr.StartSymbol().StartState() != 1 || gr.StartSymbol().EndState() != 0 || gr.StartSymbol().OnStack().front() != g.StackSymbolIndex("square")){
      std::stringstream errstr;
      errstr << "start symbol was " << gr.StartSymbol().Str() << ", but (2, [square]) was expected" << std::endl;
      throw Exception(name, errstr.str(), 1003);
    }
  }
   catch(Exception e){
  }
  
  TestEnd(name);
}

/* *****************
 * TestSp2LrProductions
 * *****************/
void TestSp2LrProductions(){
  std::string name = "Sp2Lr Productions";
  TestStart(name);
  
  PushdownGenerator g = TestGenerator8();
  
  Grammar gr = Sp2Lr(g);
  
  std::vector<Idx> dot, lambda, square;
  dot.push_back(g.StackSymbolIndex("dot"));
  square.push_back(g.StackSymbolIndex("square"));
  lambda.push_back(g.StackSymbolIndex("lambda"));
  
  Nonterminal* nt1dot = new Nonterminal(1,dot);
  GrammarSymbolPtr nt1dotPtr(nt1dot);
  Nonterminal* nt1square = new Nonterminal(1,square);
  GrammarSymbolPtr nt1squarePtr(nt1square);
  Nonterminal* nt2dot = new Nonterminal(2,dot);
  GrammarSymbolPtr nt2dotPtr(nt2dot);
  Nonterminal* nt2square = new Nonterminal(2,square);
  GrammarSymbolPtr nt2squarePtr(nt2square);
  Nonterminal* nt1dot1 = new Nonterminal(1,dot,1);
  GrammarSymbolPtr nt1dot1Ptr(nt1dot1);
  Nonterminal* nt1dot2 = new Nonterminal(1,dot,2);
  GrammarSymbolPtr nt1dot2Ptr(nt1dot2);
  Nonterminal* nt1square1 = new Nonterminal(1,square,1);
  GrammarSymbolPtr nt1square1Ptr(nt1square1);
  Nonterminal* nt1square2 = new Nonterminal(1,square,2);
  GrammarSymbolPtr nt1square2Ptr(nt1square2);
  Nonterminal* nt2dot1 = new Nonterminal(2,dot,1);
  GrammarSymbolPtr nt2dot1Ptr(nt2dot1);
  Nonterminal* nt2dot2 = new Nonterminal(2,dot,2);
  GrammarSymbolPtr nt2dot2Ptr(nt2dot2);
  Nonterminal* nt2square1 = new Nonterminal(2,square,1);
  GrammarSymbolPtr nt2square1Ptr(nt2square1);
  Nonterminal* nt2square2 = new Nonterminal(2,square,2);
  GrammarSymbolPtr nt2square2Ptr(nt2square2);
  Terminal* ta = new Terminal(g.EventIndex("a"));
  GrammarSymbolPtr taPtr(ta);
  Terminal* tlambda = new Terminal(g.EventIndex(FAUDES_PD_LAMBDA));
  GrammarSymbolPtr tlambdaPtr(tlambda);
  
  
  GrammarSymbolVector v;
  //expected read set
  std::set<GrammarProduction> read;
  v.clear();
  v.push_back(taPtr);
  v.push_back(nt1dot1Ptr);
  read.insert(GrammarProduction(*nt2dot1,v));
  v.clear();
  v.push_back(taPtr);
  v.push_back(nt1dot2Ptr);
  read.insert(GrammarProduction(*nt2dot2,v));
  v.clear();
  v.push_back(taPtr);
  v.push_back(nt1dotPtr);
  read.insert(GrammarProduction(*nt2dot,v));
  
  //expected pop set
  std::set<GrammarProduction> pop;
  v.clear();
  v.push_back(tlambdaPtr);
  pop.insert(GrammarProduction(*nt1dot2,v));
  
  //expected push set
  std::set<GrammarProduction> push;
  v.clear();
  v.push_back(nt2dot1Ptr);
  v.push_back(nt1square1Ptr);
  push.insert(GrammarProduction(*nt1square1,v));
  v.clear();
  v.push_back(nt2dot2Ptr);
  v.push_back(nt2square1Ptr);
  push.insert(GrammarProduction(*nt1square1,v));
  v.clear();
  v.push_back(nt2dot1Ptr);
  v.push_back(nt1square2Ptr);
  push.insert(GrammarProduction(*nt1square2,v));
  v.clear();
  v.push_back(nt2dot2Ptr);
  v.push_back(nt2square2Ptr);
  push.insert(GrammarProduction(*nt1square2,v));
  v.clear();
  v.push_back(nt2dot1Ptr);
  v.push_back(nt1squarePtr);
  push.insert(GrammarProduction(*nt1square,v));
  v.clear();
  v.push_back(nt2dot2Ptr);
  v.push_back(nt2squarePtr);
  push.insert(GrammarProduction(*nt1square,v));
  v.clear();
  v.push_back(nt2dotPtr);
  push.insert(GrammarProduction(*nt1square,v));
  
  //expected final state set
  std::set<GrammarProduction> final;
  v.clear();
  v.push_back(tlambdaPtr);
  final.insert(GrammarProduction(*nt2dot,v));
  v.clear();
  v.push_back(tlambdaPtr);
  final.insert(GrammarProduction(*nt2square,v));
  

  try{
    
    std::set<GrammarProduction> gp = gr.GrammarProductions();
    std::set<GrammarProduction>::const_iterator gpit;
    //test for read
    for(gpit = read.begin(); gpit != read.end(); gpit++){
      if(gp.erase(*gpit) == 0){
        std::stringstream errstr;
        errstr << "grammar production " << gpit->Str() << " , which is generated by a read transition, was expected but not found in the grammar" << std::endl;
        throw Exception(name, errstr.str(), 1003);
      }
    }
    
    //test for pop
    for(gpit = pop.begin(); gpit != pop.end(); gpit++){
      if(gp.erase(*gpit) == 0){
        std::stringstream errstr;
        errstr << "grammar production " << gpit->Str() << " , which is generated by a pop transition, was expected but not found in the grammar" << std::endl;
        throw Exception(name, errstr.str(), 1003);
      }
    }
    
    //test for push
    for(gpit = push.begin(); gpit != push.end(); gpit++){
      if(gp.erase(*gpit) == 0){
        std::stringstream errstr;
        errstr << "grammar production " << gpit->Str() << " , which is generated by a push transition, was expected but not found in the grammar" << std::endl;
        throw Exception(name, errstr.str(), 1003);
      }
    }
    
    //test for final states
    for(gpit = final.begin(); gpit != final.end(); gpit++){
      if(gp.erase(*gpit) == 0){
        std::stringstream errstr;
        errstr << "grammar production " << gpit->Str() << " , which is generated by a final transition, was expected but not found in the grammar" << std::endl;
        throw Exception(name, errstr.str(), 1003);
      }
    }
    
    //all productions must have been looked at by now
    if(gp.size() != 0){
      std::stringstream errstr;
      errstr << "the grammar contained" << gp.size() << " more productions than expected" << std::endl;
      throw Exception(name, errstr.str(), 1003);
    }
  }
   catch(Exception e){
  }
  
  TestEnd(name);
}

/* *****************
 * TestSp2Lr2Productions
 * *****************/
void TestSp2Lr2Productions(){
  std::string name = "Sp2Lr2 Productions";
  TestStart(name);
  
  PushdownGenerator g = TestGenerator8();
  
  Grammar gr = Sp2Lr2(g);
  
  std::vector<Idx> dot, lambda, square;
  dot.push_back(g.StackSymbolIndex("dot"));
  square.push_back(g.StackSymbolIndex("square"));
  lambda.push_back(g.StackSymbolIndex("lambda"));
  
  Nonterminal* nt1dot = new Nonterminal(1,dot);
  GrammarSymbolPtr nt1dotPtr(nt1dot);
  Nonterminal* nt1square = new Nonterminal(1,square);
  GrammarSymbolPtr nt1squarePtr(nt1square);
  Nonterminal* nt2dot = new Nonterminal(2,dot);
  GrammarSymbolPtr nt2dotPtr(nt2dot);
  Nonterminal* nt2square = new Nonterminal(2,square);
  GrammarSymbolPtr nt2squarePtr(nt2square);
  Nonterminal* nt1dot1 = new Nonterminal(1,dot,1);
  GrammarSymbolPtr nt1dot1Ptr(nt1dot1);
  Nonterminal* nt1dot2 = new Nonterminal(1,dot,2);
  GrammarSymbolPtr nt1dot2Ptr(nt1dot2);
  Nonterminal* nt1square1 = new Nonterminal(1,square,1);
  GrammarSymbolPtr nt1square1Ptr(nt1square1);
  Nonterminal* nt1square2 = new Nonterminal(1,square,2);
  GrammarSymbolPtr nt1square2Ptr(nt1square2);
  Nonterminal* nt2dot1 = new Nonterminal(2,dot,1);
  GrammarSymbolPtr nt2dot1Ptr(nt2dot1);
  Nonterminal* nt2dot2 = new Nonterminal(2,dot,2);
  GrammarSymbolPtr nt2dot2Ptr(nt2dot2);
  Nonterminal* nt2square1 = new Nonterminal(2,square,1);
  GrammarSymbolPtr nt2square1Ptr(nt2square1);
  Nonterminal* nt2square2 = new Nonterminal(2,square,2);
  GrammarSymbolPtr nt2square2Ptr(nt2square2);
  Terminal* ta = new Terminal(g.EventIndex("a"));
  GrammarSymbolPtr taPtr(ta);
  Terminal* tlambda = new Terminal(g.EventIndex(FAUDES_PD_LAMBDA));
  GrammarSymbolPtr tlambdaPtr(tlambda);
  
  
  GrammarSymbolVector v;
  //expected read set
  std::set<GrammarProduction> read;
  v.clear();
  v.push_back(taPtr);
  v.push_back(nt1dot2Ptr);
  read.insert(GrammarProduction(*nt2dot2,v));
  
  //expected pop set
  std::set<GrammarProduction> pop;
  v.clear();
  v.push_back(tlambdaPtr);
  pop.insert(GrammarProduction(*nt1dot2,v));
  
  //expected push set
  std::set<GrammarProduction> push;
  v.clear();
  v.push_back(nt2dot2Ptr);
  v.push_back(nt2squarePtr);
  push.insert(GrammarProduction(*nt1square,v));
  v.clear();
  v.push_back(nt2dotPtr);
  push.insert(GrammarProduction(*nt1square,v));
  
  //expected final state set
  std::set<GrammarProduction> final;
  v.clear();
  v.push_back(tlambdaPtr);
  final.insert(GrammarProduction(*nt2dot,v));
  v.clear();
  v.push_back(tlambdaPtr);
  final.insert(GrammarProduction(*nt2square,v));

  try{
    
    std::set<GrammarProduction> gp = gr.GrammarProductions();
    std::set<GrammarProduction>::const_iterator gpit;
    //test for read
    for(gpit = read.begin(); gpit != read.end(); gpit++){
      if(gp.erase(*gpit) == 0){
        std::stringstream errstr;
        errstr << "grammar production " << gpit->Str() << " , which is generated by a read transition, was expected but not found in the grammar" << std::endl;
        throw Exception(name, errstr.str(), 1003);
      }
    }
    
    //test for pop
    for(gpit = pop.begin(); gpit != pop.end(); gpit++){
      if(gp.erase(*gpit) == 0){
        std::stringstream errstr;
        errstr << "grammar production " << gpit->Str() << " , which is generated by a pop transition, was expected but not found in the grammar" << std::endl;
        throw Exception(name, errstr.str(), 1003);
      }
    }
    
    //test for push
    for(gpit = push.begin(); gpit != push.end(); gpit++){
      if(gp.erase(*gpit) == 0){
        std::stringstream errstr;
        errstr << "grammar production " << gpit->Str() << " , which is generated by a push transition, was expected but not found in the grammar" << std::endl;
        throw Exception(name, errstr.str(), 1003);
      }
    }
    
    //test for final states
    for(gpit = final.begin(); gpit != final.end(); gpit++){
      if(gp.erase(*gpit) == 0){
        std::stringstream errstr;
        errstr << "grammar production " << gpit->Str() << " , which is generated by a final transition, was expected but not found in the grammar" << std::endl;
        throw Exception(name, errstr.str(), 1003);
      }
    }
    
    //all productions must have been looked at by now
    if(gp.size() != 0){
      std::stringstream errstr;
      errstr << "the grammar contained" << gp.size() << " more productions than expected" << std::endl;
      throw Exception(name, errstr.str(), 1003);
    }
  }
   catch(Exception e){
  }
  
  TestEnd(name);
}

/* *****************
 * TestRupProductions
 * *****************/
void TestRupProductions(){
  std::string name = "Test Rup Productions";
  TestStart(name);
  
  Grammar gr = TestGrammar3();
  Grammar rGr = Rup(gr);
  
  try{
    
    //only one production must have been removed
    if(gr.GrammarProductions().size() - 1 != rGr.GrammarProductions().size()){
      std::stringstream errstr;
      errstr << "size of grammar productions was " << rGr.GrammarProductions().size() << ", but " << gr.GrammarProductions().size() - 1 << " was expected" << std::endl;
      throw Exception(name, errstr.str(), 1003);
    }
    
    //only the grammar production (2, [dot], 2) -> a(1, [dot]) must have been removed
    std::vector<Idx> v;
    v.push_back(PushdownGenerator::GlobalStackSymbolTablep()->Index("dot"));
    Nonterminal* nt1 = new Nonterminal(2,v,2);
    GrammarSymbolPtr nt1Ptr(nt1);
    Nonterminal* nt2 = new Nonterminal(1,v);
    GrammarSymbolPtr nt2Ptr(nt2);
    Terminal* t = new Terminal(PushdownGenerator::GlobalEventSymbolTablep()->Index("a"));
    GrammarSymbolPtr tPtr(t);
    GrammarSymbolVector gs;
    gs.push_back(tPtr);
    gs.push_back(nt2Ptr);
    GrammarProduction gp(*nt1,gs);
    
    if(gr.GrammarProductions().find(gp) != gr.GrammarProductions().end()){
      std::stringstream errstr;
      errstr << "grammar production (2, [dot], 2) -> a(1, [dot]) was present, but was expected to be deleted" << std::endl;
      throw Exception(name, errstr.str(), 1003);
    }
  }
   catch(Exception e){
  }
  
  TestEnd(name);
}

/* *****************
 * TestRupNonterminals
 * *****************/
void TestRupNonterminals(){
  std::string name = "Test Rup Nonterminals";
  TestStart(name);
  
  Grammar gr = TestGrammar3();
  Grammar rGr = Rup(gr);
  
  try{
    
    //only one nonterminal must have been removed
    if(gr.Nonterminals().size() - 1 != rGr.Nonterminals().size()){
      std::stringstream errstr;
      errstr << "size of nonterminals was " << rGr.Nonterminals().size() << ", but " << gr.Nonterminals().size() - 1 << " was expected" << std::endl;
      throw Exception(name, errstr.str(), 1003);
    }
    
    //only the nonterminal (2, [dot], 2) must have been removed
    std::vector<Idx> v;
    v.push_back(PushdownGenerator::GlobalStackSymbolTablep()->Index("dot"));
    Nonterminal nt(2,v,2);
    if(gr.Nonterminals().find(nt) != gr.Nonterminals().end()){
      std::stringstream errstr;
      errstr << "nonterminal (2, [dot], 2) was present, but was expected to be deleted" << std::endl;
      throw Exception(name, errstr.str(), 1003);
    }
  }
   catch(Exception e){
  }

  
  TestEnd(name);
}

/* *****************
 * TestFilter
 * *****************/
void TestFilter(){
  TestFilterMixedGrammarSymbols();
  TestFilterNothing();
}

/* *****************
 * TestRnpp1
 * *****************/
void TestRnpp1(){
  TestRnpp1FindSymbolsEmptySet();
  TestRnpp1FindSymbolsNonemptySet();
}

/* *****************
 * TestRnpp1
 * *****************/
void TestRnppl(){
  TestRnpplFindSymbolsEmptySet();
  TestRnpplFindSymbolsNonemptySet();
  TestRnpplFindSymbolsCompleteSet();
}

/* *****************
 * TestRnpp
 * *****************/
void TestRnpp(){
  TestRnppGrammar1();
  TestRnppGrammar2();
  TestRnppEmptyGrammar();
}

/* *****************
 * Sp2Lr
 * *****************/
void TestSp2Lr(){
  
  TestSp2LrTerminals();
  TestSp2LrNonterminals();
  TestSp2LrProductions();
}

/* *****************
 * Sp2Lr2
 * *****************/
void TestSp2Lr2(){
  TestSp2Lr2Productions();
}

/* *****************
 * Rup
 * *****************/
void TestRup(){
  
  TestRupProductions();
  TestRupNonterminals();
}
} // namespace faudes

