/** @file cfl_statemin.cpp state space minimization */

/* FAU Discrete Event Systems Library (libfaudes)

Copyright (C) 2006  Bernd Opitz
Exclusive copyright is granted to Klaus Schmidt

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA */

  
#include "cfl_statemin.h"
#include "cfl_exception.h"
#include "cfl_project.h"

namespace faudes {

// StateMin(rGen, rResGen)
void StateMin(const Generator& rGen, Generator& rResGen) {
  // how to avoid the copy?? see core procedure
  Generator* accgen=rGen.Copy();
  StateMin(*accgen, rResGen);
}

// StateMin(rGen, rResGen)
void aStateMin(const Generator& rGen, Generator& rResGen) {
  Generator* accgen=rGen.Copy();
  StateMin(*accgen, rResGen);
  // restore attributes
  rResGen.EventAttributes(accgen->Alphabet());
}

// StateMin(rGen)
void aStateMin(Generator& rGen) {
  Generator* accgen=rGen.Copy();
  StateMin(*accgen, rGen);
  // restore attributes
  rGen.EventAttributes(accgen->Alphabet());
}

// StateMin(rGen, rResGen)
void StateMin(Generator& rGen, Generator& rResGen) {
  std::vector<StateSet> subsets;
  std::vector<Idx> newindices;
  StateMin(rGen, rResGen, subsets, newindices);
}

// StateMin(rGen, rResGen, rSubSets, rNewIndices)
// todo: make argument const, circumvent accessible
void StateMin(Generator& rGen, Generator& rResGen, 
	      std::vector<StateSet>& rSubsets, std::vector<Idx>& rNewIndices) {
  FD_DF("StateMin: *** computing state space minimization of generator " 
	<< &rGen << " ***");

  FD_DF("StateMin: making generator accessible");
  rGen.Accessible();
  if (rGen.Size() == 0) {
    FD_DF("StateMin: generator size 0. returning given generator");
    rResGen = rGen;
    return;
  }
  if(rGen.Size() == 1) {
    FD_DF("StateMin: generator size 1. returning given generator");
    rResGen = rGen;
    rSubsets.push_back(rGen.States() );
    rNewIndices.push_back(*( rResGen.States().Begin() ) );
    return;
  }

  // ensure generator is deterministic
#ifdef FAUDES_CHECKED
  if(!rGen.IsDeterministic()) {
    throw Exception("StateMin", "input automaton nondeterministic", 101);
  }
#endif
    
  // use pointer pResGen to result rResGen; if rResGen is identical to
  // one of the parameters, allocate temporary object and copy back later
  Generator* pResGen = &rResGen;
  if(&rResGen==&rGen) {
    pResGen= pResGen->New();
  }

  // prepare result
  pResGen->Clear();
  pResGen->Name(rGen.Name()+" [minstate]");
  pResGen->InjectAlphabet(rGen.Alphabet());
  bool stateNames= pResGen->StateNamesEnabled() && rGen.StateNamesEnabled();
  // blocks B[i]
  std::vector<StateSet>& b = rSubsets; // convenience notation "b"
  Idx i, j;
  // set of active b (iterators)
  std::set<Idx> active;
  std::set<Idx>::iterator ait;
  // reverse gen transrel
  TransSetEvX2X1 rtransrel;
  rGen.TransRel(rtransrel);
  TransSetEvX2X1::Iterator rtit, rtit_end;
  // other stuff
  StateSet::Iterator sit; 
  TransSet::Iterator tit, tit_end;

  // set up b "B[i]"
  b.clear();
  i = 0;
  if (rGen.Size() - rGen.MarkedStatesSize() > 0) {
    StateSet notmarked=rGen.States() - rGen.MarkedStates();
    notmarked.Name("B[i]"); 
    FD_DF("StateMin: new block B[" << i << "] = {" << notmarked.ToString() << "}");
    b.push_back(notmarked);  //X -Xm
    active.insert(i++);
  }
  if (rGen.MarkedStatesSize() > 0) { // tmoor 200909: conditional to prevent emty block 
    FD_DF("StateMin: new block B[" << i << "] = {" << rGen.MarkedStates().ToString() << "}");
    b.push_back(rGen.MarkedStates()); // Xm
    active.insert(i++);
  }

  // while there is a active block B
  while (! active.empty()) {
    FD_WPC(b.size()-active.size(), b.size(), "StateMin: blocks/active:   " << b.size() << " / " << active.size());
#ifdef FAUDES_DEBUG_FUNCTION
    FD_DF("StateMin: while there is an active block B...");
    std::set<Idx>::iterator _it1;
    std::stringstream str;
    for (_it1 = active.begin(); _it1 != active.end(); ++_it1) {
      str << *_it1 << " ";
    }
    FD_DF("StateMin: active: "+str.str());
    std::vector<StateSet>::iterator _it2;
    str.clear();
    str.str("");
    for (_it2 = b.begin(); _it2 != b.end(); ++_it2) {
      str << "{" << _it2->ToString() << "} "<<std::endl;
    }
    str << std::endl;
    FD_DF("B: "+str.str());
#endif
    // current block B[i]
    i = *(active.begin());
    // inactivate B[i]
    active.erase(active.begin());
    FD_DF("StateMin: getting active block B[" << i << "] = {" <<
	  b.at(i).ToString() << "}");
    // b_current <- B[i]
    StateSet b_current = b.at(i);
 
    // compute C = f^-1(B[i]) for every event in B[i] (as b_current)
    StateSet c;
    EventSet::Iterator eit;
    // iteration over alphabet
    for (eit = rGen.AlphabetBegin(); eit != rGen.AlphabetEnd(); ++eit) {
      c.Clear();
      // iteration over states in current block
      for (sit = b_current.Begin(); sit != b_current.End(); ++sit) {
	// find predecessor states by current ev + x2
	rtit = rtransrel.BeginByEvX2(*eit, *sit);
	rtit_end = rtransrel.EndByEvX2(*eit, *sit);
	for (; rtit != rtit_end; ++rtit) {
	  c.Insert(rtit->X1);
	}
      }
      // if no predecessor states where found, try next event
      if(c.Empty()) continue;
      // search for block to be split 
      FD_DF("StateMin: computed predecessor states C = {" << c.ToString() 
	      << "} for event " << rGen.EventName(*eit));
      // foreach block D 
      for (j=0; j < b.size(); ++j) {
        // d_current <- B[j]
        const StateSet& d_current = b.at(j);
	FD_DF("StateMin: examining block B[" << j << "] = {" << 
		d_current.ToString() << "}");
	// compute D' = D intersection C
	StateSet d_ = d_current * c;
	d_.Name("D'");
	// check D split by B
	if(d_.Empty() || (d_.Size()==d_current.Size())) {
	  FD_DF("StateMin: -> no split");  
	  continue;
	}
	FD_DF("StateMin: -> split:");  
	// compute D'' = D intersected not C; 
	StateSet d__ = d_current - d_;
	d__.Name("D''");
     	// record split block
	b[j] = d_;
	b.push_back(d__);
	FD_DF("StateMin: new block B[" << j << "] = {" << d_.ToString() << "}");
	FD_DF("StateMin: new block B[" << b.size()-1 << "] = {" << d__.ToString() << "}");
  	// if B[j] was active then mark both D', D'' active
	if(active.find(j) != active.end()) {
	  active.insert((Idx)b.size()- 1);
	  FD_DF("StateMin: mark active: " << b.size()-1);
	}
	// else mark smaller of D', D'' active
	else {
	  if (d_.Size() < d__.Size()) {
	    active.insert(j);
	    FD_DF("StateMin: mark active: " << j);
	  } else {
	    active.insert((Idx)b.size()-1);
	    FD_DF("StateMin: mark active: " << b.size()-1);
	  }
	}
      } // foreach block D
    } // foreach event
  } // while active blocks exist

  FD_DF("StateMin: *** building minimized generator ***");
  // build minimized generator
  std::map<Idx,Idx> minstatemap;
  Idx newstate;
  // loop over all blocks B
  for (i = 0; i < b.size(); ++i) {
    // create state in new generator for every block
    newstate = pResGen->InsState();
    rNewIndices.push_back(newstate);
    FD_DF("StateMin: block {" << b.at(i).ToString() 
	  << "} -> new state " << newstate);
    std::ostringstream ostr; 
    for (sit = b.at(i).Begin(); sit != b.at(i).End(); ++sit) {
      // set minstatemap entry for every state in gen
      minstatemap[*sit] = newstate;
      if(stateNames) {
	if (rGen.StateName(*sit) == "") {
	  ostr << ToStringInteger(*sit) << ",";
	}
	else {
	  ostr << rGen.StateName(*sit) << ",";
	}
      }
      // set istates
      if (rGen.ExistsInitState(*sit)) {
	pResGen->SetInitState(newstate);
	FD_DF("StateMin: -> initial state");
      }
      // set mstates
      if (rGen.ExistsMarkedState(*sit)) {
	pResGen->SetMarkedState(newstate);
	FD_DF("StatMmin: -> marked state");
      }
    }
    if(stateNames) {
      std::string statename = ostr.str();
      if(statename!="") statename.erase(statename.length()-1);
      statename = "{" + statename + "}";
      pResGen->StateName(newstate, statename);
    }
  }
  // create transition relation
  for (tit = rGen.TransRelBegin(); tit != rGen.TransRelEnd(); ++tit) {
    pResGen->SetTransition(minstatemap[tit->X1], tit->Ev, minstatemap[tit->X2]);
    FD_DF("statemin: adding transition: " 
	  << minstatemap[tit->X1] << "-" << tit->Ev << "-" 
	  << minstatemap[tit->X2]);
  }
    
  // if necessary, move pResGen to rResGen
  if(pResGen != &rResGen) {
    pResGen->Move(rResGen);
    delete pResGen;
  }
}
  
} // namespace faudes
