/** @file pd_pdgenerator.h  pushdown generator class TpdGenerator */

/* Pushdown plugin for FAU Discrete Event Systems Library (libfaudes)

   Copyright (C) 2013  Stefan Jacobi, Sven Schneider, Anne-Kathrin Hess

*/

#ifndef FAUDES_PD_PDGENERATOR_H
#define FAUDES_PD_PDGENERATOR_H
#define FAUDES_DEBUG_GENERATOR
#define FAUDES_DEBUG_CONTAINER

#include "corefaudes.h"
#include "pd_attributes.h"
#include "pd_grammar.h"


namespace faudes {



/**
 * Generator with push down extensions. 
 *
 *
 * @ingroup PushdownPlugin
 * 
 * @section Overview
 *  tratarterteartawrtrae
 * 
 * @section Contents
 *  fvdgdthrthorthtiop
 */

template <class GlobalAttr, class StateAttr, class EventAttr, class TransAttr>
class TpdGenerator : public TcGenerator<GlobalAttr, StateAttr, EventAttr, TransAttr> {  
	
public:
	/**
	 * Constructor
	 */
	TpdGenerator(void);
	
	/** 
	 * Copy constructor 
	 *
	 * @param rOtherGen
	 */
	TpdGenerator(const TpdGenerator& rOtherGen);
	
	/** 
	 * Copy constructor (no attributes)
	 *
	 * @param rOtherGen
	 */
	TpdGenerator(const vGenerator& rOtherGen);
	
	/**
	 * Assignment operator (uses copy)
	 *  Note: you must reimplement this operator in derived 
	 *  classes in order to handle internal pointers correctly
	 *
	 * @param rOtherGen
	 *   Other generator
	 */
	virtual TpdGenerator& operator= (const TpdGenerator& rOtherGen) {this->Assign(rOtherGen); return *this;};

	/**
	 * Assignment operator (uses copy)
	 *
	 * @param rOtherGen
	 *   Other generator
	 */
	virtual TpdGenerator& operator= (const vGenerator& rOtherGen) {this->Assign(rOtherGen); return *this;};
	
	/**
	 * Construct from file
	 *
	 * @param rFileName
	 *   Name of file
	 *
	 * @exception Exception
	 *   - file format errors (id 1, 50, 51, 52)
	 */
	TpdGenerator(const std::string& rFileName);
	
	/**
	 * Construct on heap. 
	 * Constructs a TpdGenerator on heap.
	 *
	 * @return 
	 *   new Generator 
	 */
	TpdGenerator* New(void) const;
	
	/**
	 * Construct copy on heap. 
	 * Constructs a TpdGenerator on heap.
	 *
	 * @return 
	 *   new Generator 
	 */
	TpdGenerator* Copy(void) const;
	
	/**
	 * Type test.
	 * Uses C++ dynamic cast to test whether the specified object
	 * casts to a PushdownGenerator.
	 *
	 * @return 
	 *   TpdGenerator reference if dynamic cast succeeds, else NULL 
	 */
	virtual const Type* Cast(const Type* pOther) const {
		return dynamic_cast< const TpdGenerator* > (pOther);
	};
	
	
	/**
	 * Construct on stack.
	 * Constructs a TpdGenerator on stack.
	 *
	 * @return 
	 *   new Generator 
	 */
	TpdGenerator NewPdGen(void) const;
	
    /**
    * Get Pointer to global StackSymbolTable. This is
    * a static member of SymbolTable and used as default
    * for all derived generator classes and instantiated objects.
    *
    * @return
    *   Pointer to global StackSymbolTable
    */
    static SymbolTable* GlobalStackSymbolTablep(void);
    
    
    /**
    * Get Pointer to mpStackSymbolTable.
    *
    * @return Pointer mpStackSymbolTable
    */
    SymbolTable* StackSymbolTablep(void) const;

   /**
    * Set StackSymbolTable.
    *
    * @param pStackSymTab
    *    Pointer SymbolTable
    */
    void StackSymbolTablep(SymbolTable* pStackSymTab);
	
    /**
     * Return a NameSet with generator's StackSymbolTable 
     *
     * @return
     *   New empty StackSymbolSet on stack
     */
     StackSymbolSet NewStackSymbolSet(void) const;

    /**
     * Construct a stack symbol on heap.
     *
     * @return
     *   Pointer to new empty StackSymbolSet on heap
     */
     StackSymbolSet* NewStackSymbolSetp(void) const;

   /**
    * Number of stacks symbols in mStackSymbols
    *
    * @return Number of stack symbols in mStackSymbols
    */
    Idx StackSymbolsSize(void) const;

   /**
    * Get stack symbol set as const reference
    *
    * @return mStackSymbols
    */
    const StackSymbolSet& StackSymbols(void) const;


   /**
    * Get stack symbol set as pointer
    *
    * @return mStackSymbols
    */
    StackSymbolSet* StackSymbolsp(void);

   /**
    * Overwrites mStackSymbols with new stack symbols without consistency check
    *
    * @param newstacksymbols
    *    New stack symbols that are written to mStackSymbols
    */
    void InjectStackSymbols(const StackSymbolSet& newstacksymbols);

   /**
    * Looks up stack symbol name for given index
    *
    * @param index
    *    Stack symbol index
    *
    * @return Stack symbol name
    */
    std::string StackSymbolName(Idx index) const;
    
    /**
    * Returns stack symbol for given index
    *
    * @param index
    *    Stack symbol index
    *
    * @return Stack symbol object
    */
    StackSymbol StackSymbolObj(Idx index) const;
    
    /**
    * Looks up stack symbol index for given name
    *
    * @param rName
    *    Stack symbol name
    *
    * @return Stack symbol index or 0 for nonexistent
    */
    Idx StackSymbolIndex(const std::string& rName) const;

    /**
    * Add named stack symbol to generator. An entry in the mpStackSymbolTable will
    * be made if stack symbol is new.
    *
    * @param rName
    *   Name of the stack symbol to add
    *
    * @return 
    *   New unique index
    */
    Idx InsStackSymbol(const std::string& rName);
    
    /**
    * Add stack symbol to generator. An entry in the mpStackSymbolTable will
    * be made if stack symbol is new.
    *
    * @param rSymbol
    *   The stack symbol to add
    *
    * @return 
    *   New unique index
    */
    Idx InsStackSymbol(const StackSymbol& rSymbol);
    
    /**
    * Add named stack bottom to generator. An entry in the mpStackSymbolTable will
    * be made if stack symbol is new. This will replace any old stack bottom symbol.
    *
    * @param rName
    *   Name of the stack bottom symbol to add
    *
    * @return 
    *   Stack bottom symbol index
    */
    Idx SetStackBottom(const std::string& rName);
    
    /**
    * Add stack bottom to generator. An entry in the mpStackSymbolTable will
    * be made if stack symbol is new. This will replace any old stack bottom symbol.
    *
    * @param rSymbol
    *   The stack bottom symbol to add
    *
    * @return 
    *   Stack bottom symbol index
    */
    Idx SetStackBottom(const StackSymbol& rSymbol);
    
    /**
    * Add stack bottom to generator. An entry in the mpStackSymbolTable will
    * be made if stack symbol is new. This will replace any old stack bottom symbol.
    *
    * @param idx
    *   The index of the stack bottom symbol to add
    *
    * @return 
    *   Stack bottom symbol index
    */
    Idx SetStackBottom(const Idx idx);
    
    /**
     * Get the index of the stack bottom symbol
     * 
     * @return
     *      index of the stack bottom symbol
     */
    Idx StackBottom() const;
    
    /**
    * Add new named stack symbols to generator.
    *
    * @param rStackSymbolSet
    *   StackSymbolSet
    */
    void InsStackSymbols(const StackSymbolSet& rStackSymbolSet);

    /**
    * Delete stack symbol from generator by index.
    *
    * @param index
    *   Index of stack symbol
    * @return
    *   True if stack symbol did exist
    *
    */
    bool DelStackSymbol(Idx index);
    
    /**
    * Delete stack symbol from generator by name. mpStackSymbolTable stays untouched.
    *
    * @param rName
    *    Name of stack symbol
    * @return
    *    True if stack symbol did exist
    */
    bool DelStackSymbol(const std::string& rName);
    
    /** 
    * Delete a set of stack symbols from generator. 
    * 
    * @param rStackSymbols
    *   StackSymbolSet containing stack symbols to remove
    */
    void DelStackSymbols(const StackSymbolSet& rStackSymbols);
    
   /** 
    * Test existence of stack symbol in mStackSymbols
    *
    * @param index
    *   Stack symbol index
    *
    * @return
    *   true / false
    */
    bool ExistsStackSymbol(Idx index) const;

   /** 
    * Test existence of stack symbol in mStackSymbols
    *
    * @param rName
    *   Stack symbol name
    *
    * @return
    *   True if stack symbol exists
    */
    bool ExistsStackSymbol(const std::string& rName) const;

   /**
    * Returns an iterator to stack symbol index in mStackSymbols
    *
    * @param index
    *   Index to find
    *
    * @return
    *   StackSymbolSet::Iterator to stack symbol index
    */
    StackSymbolSet::Iterator FindStackSymbol(Idx index) const;

   /**
    * Returns an iterator to stack symbol index in mStackSymbols
    *
    * @param rName
    *   Stack symbol name of index to find
    *
    * @return
    *   StackSymbolSet::Iterator to stack symbol index
    */
    StackSymbolSet::Iterator FindStackSymbol(const std::string& rName) const;

   /**
    * Iterator to Begin() of mStackSymbols
    *
    * @return iterator to begin of mStackSymbols
    */
    StackSymbolSet::Iterator StackSymbolsBegin(void) const;
    
    /**
    * Iterator to End() of mStackSymbols
    *
    * @return iterator to end of mStackSymbols
    */
    StackSymbolSet::Iterator StackSymbolsEnd(void) const;
    
    /**
     * Converts a vector of stack symbols to a vector of Idx
     * 
     * @param symbols
     *      the vector of stack symbols
     * @return
     *      vector of Idx
     */
    std::vector<Idx> StackSymbolsToIndices(const std::vector<StackSymbol> symbols) const;
    
    
    /**
     * Get number of transitions including pop-push-pairs
     * 
     * @return
     *      number of transitions
     */
    Idx TransRelSize() const;
    
	/** 
	 * Add a transition to generator by indices. States and event
	 * must already exist!
	 *
	 * Define FAUDES_CHECKED for consistency checks.
	 *
	 * @param x1 
	 *   Predecessor state index 
	 * @param ev 
	 *   Event index 
	 * @param x2
	 *   Successor state index
	 *
	 * @return
	 *   True, if the transition was new the generator
	 *
	 * @exception Exception
	 *   - state or event not in generator (id 95)
	 */
	bool SetTransition(Idx x1, Idx ev, Idx x2);
	
	/** 
	 * Add a transition to generator by names. Statename and eventname
	 * must already exist!
	 *
	 * @param rX1
	 *   Predecessor state name
	 * @param rEv
	 *   Event name
	 * @param rX2
	 *   Successor state name
	 *
	 * @return
	 *   True, if the transition was new the generator
	 *
	 * @exception Exception
	 *   - state or event not in generator (id 95)
	 *   - state name not known (id 90)
	 *   - event name not known (id 66)
	 */
	bool SetTransition(const std::string& rX1, const std::string& rEv, 
					   const std::string& rX2);
	
	/** 
	 * Add a transition with attribute to generator. States and event
	 * must already exist!
	 *
	 * Define FAUDES_CHECKED for consistency checks.
	 *
	 * @param rTransition
	 *   transition
	 * @param rAttr
	 *   attribute
	 *
	 * @return
	 *   True, if the transition was new the generator
	 *
	 */
	bool SetTransition(const Transition& rTransition, const TransAttr& rAttr);
	
	/**
	 * Inserts new PushdownTransition constructed from parameters.
	 * Performs consistency checks for x1, x2, ev and all stack symbols in rPop and rPush.
	 *
	 * @param rTrans
	 *    new transition 
	 * @param rPop
	 *    Stack symbol vector to be popped when transitioning
	 * @param rPush
	 *    Stack symbol vector to be pushed when transitioning
	 * @return
	 *   True, if the transition was new to the generator
	 */
	bool SetTransition(const Transition& rTrans, const std::vector<StackSymbol>& rPop, const std::vector<StackSymbol>& rPush);
    
    
    /**
     * Inserts new PushdownTransition constructed from parameters.
     * Performs consistency checks for x1, x2, ev and all stack symbols in rPop and rPush.
     *
     * @param rTrans
     *    new transition 
     * @param rPop
     *    Stack symbol vector to be popped when transitioning
     * @param rPush
     *    Stack symbol vector to be pushed when transitioning
     * @return
     *   True, if the transition was new to the generator
     */
    bool SetTransition(const Transition& rTrans, const std::vector<Idx>& rPop, const std::vector<Idx>& rPush);

	
    /**
     * Inserts new PushdownTransition constructed from parameters.
     * Performs consistency checks for x1, x2, ev and stack symbols in rPop and rPush.
     *
     * @param x1
     *    Start state of new PushdownTransition.
     * @param ev
     *    Event of new PushdownTransition.
     * @param x2
     *    End state of new PushdownTransition.
     * @param rPop
     *    Stack symbol vector to be popped when transitioning
     * @param rPush
     *    Stack symbol vector to be pushed when transitioning
     * @return
     *   True, if the transition was new to the generator
     */
    bool SetTransition(Idx x1, Idx ev, Idx x2,
              const std::vector<StackSymbol>& rPop, const std::vector<StackSymbol>& rPush);
    
    /**
     * Inserts new PushdownTransition constructed from parameters.
     * Performs consistency checks for x1, x2, ev and stack symbols in rPop and rPush.
     *
     * @param x1
     *    Start state of new PushdownTransition.
     * @param ev
     *    Event of new PushdownTransition.
     * @param x2
     *    End state of new PushdownTransition.
     * @param rPop
     *    Stack symbol vector to be popped when transitioning
     * @param rPush
     *    Stack symbol vector to be pushed when transitioning
     * @return
     *   True, if the transition was new to the generator
     */
    bool SetTransition(Idx x1, Idx ev, Idx x2,
              const std::vector<Idx>& rPop, const std::vector<Idx>& rPush);
	
    /**
     * Inserts new PushdownTransition constructed from parameters.
     * Performs consistency checks for x1, x2, ev and stack symbols in rPop and rPush.
     *
     * @param rX1
     *    Start state of new PushdownTransition.
     * @param rEv
     *    Event of new PushdownTransition.
     * @param rX2
     *    End state of new PushdownTransition.
     * @param rPop
     *    Stack symbol vector to be popped when transitioning
     * @param rPush
     *    Stack symbol vector to be pushed when transitioning
     * @return
     *   True, if the transition was new to the generator
    */
    bool SetTransition(const std::string& rX1, const std::string& rEv, const std::string& rX2, const std::vector<StackSymbol>& rPop, const std::vector<StackSymbol>& rPush);
    
    /**
     * Inserts new PushdownTransition constructed from parameters.
     * Performs consistency checks for x1, x2, ev and stack symbols in rPop and rPush.
     *
     * @param rX1
     *    Start state of new PushdownTransition.
     * @param rEv
     *    Event of new PushdownTransition.
     * @param rX2
     *    End state of new PushdownTransition.
     * @param rPop
     *    Stack symbol vector to be popped when transitioning
     * @param rPush
     *    Stack symbol vector to be pushed when transitioning
     * @return
     *   True, if the transition was new to the generator
    */
    bool SetTransition(const std::string& rX1, const std::string& rEv, const std::string& rX2, const std::vector<Idx>& rPop, const std::vector<Idx>& rPush);
    
    /**
     * Delete an exisiting PushdownTransition with the provided parameters.
     *
     * @param x1
     *    Start state of the PushdownTransition.
     * @param ev
     *    Event of the PushdownTransition.
     * @param x2
     *    End state of the PushdownTransition.
     * @param rPop
     *    Stack symbol vector to be popped when transitioning
     * @param rPush
     *    Stack symbol vector to be pushed when transitioning
     * @return
     *   True, if the transition was deleted from the generator
     */
    bool ClrTransition(Idx x1, Idx ev, Idx x2,
              const std::vector<StackSymbol>& rPop, const std::vector<StackSymbol>& rPush);
    
    /**
     * Delete an exisiting PushdownTransition with the provided parameters.
     *
     * @param x1
     *    Start state of the PushdownTransition.
     * @param ev
     *    Event of the PushdownTransition.
     * @param x2
     *    End state of the PushdownTransition.
     * @param rPop
     *    Stack symbol vector to be popped when transitioning
     * @param rPush
     *    Stack symbol vector to be pushed when transitioning
     * @return
     *   True, if the transition was deleted from the generator
     */
    bool ClrTransition(Idx x1, Idx ev, Idx x2,
              const std::vector<Idx>& rPop, const std::vector<Idx>& rPush);
    
    
    /**
     * Delete an exisiting PushdownTransition with the provided parameters.
     *
     * @param rTrans
     *    the transition
     * @param rPop
     *    Stack symbol vector to be popped when transitioning
     * @param rPush
     *    Stack symbol vector to be pushed when transitioning
     * @return
     *   True, if the transition was deleted from the generator
     */
    bool ClrTransition(const Transition& rTrans,
              const std::vector<StackSymbol>& rPop, const std::vector<StackSymbol>& rPush);
    
    /**
     * Delete an exisiting PushdownTransition with the provided parameters.
     *
     * @param rTrans
     *    the transition
     * @param rPop
     *    Stack symbol vector to be popped when transitioning
     * @param rPush
     *    Stack symbol vector to be pushed when transitioning
     * @return
     *   True, if the transition was deleted from the generator
     */
    bool ClrTransition(const Transition& rTrans,
              const std::vector<Idx>& rPop, const std::vector<Idx>& rPush);


    /**
     * Get the pop/push set attached to this transition
     * 
     * @param rTrans
     *      the transition
     * @return
     *      the pop/push set
     */
    const PopPushSet& PopPush(const Transition& rTrans) const;
    
    /**
     * Get an iterator to the beginning of the pop/push set attached to this
     * transition
     * 
     * @param rTrans
     *      the transition
     * @return
     *      iterator to the beginning of the pop/push set
     */
    PopPushSet::const_iterator PopPushBegin(const Transition& rTrans) const;
    
    /**
     * Get an iterator to the end of the pop/push set attached to this
     * transition
     * 
     * @param rTrans
     *      the transition
     * @return
     *      iterator to the end of the pop/push set
     */
    PopPushSet::const_iterator PopPushEnd(const Transition& rTrans) const;
    
     
    /**
     * determine if the stack symbol associated with the given index is lambda
     * 
     * @param index
     *      index of a stack symbol
     * @return
     *      true if the associated stack symbol is lambda, else false
     */
    bool IsStackSymbolLambda(Idx index) const;
    
    /**
     * determine if the event associated with the given index is lambda
     * 
     * @param index
     *      index of an event
     * @return
     *      true if the associated event is lambda, else false
     */
    bool IsEventLambda(Idx index) const;
	
    
   /**
    * Throw exception if stack symbol refers to stack symbol not
    * in stack symbol set
    *
    * @exception Exception
    *   - invalid stack symbol (id 1001)
    */
    void ConsistentStackSymbol(const StackSymbol& rStackSymbol) const;
    
    /**
    * Throw exception if stack symbol refers to stack symbol not
    * in stack symbol set
    *
    * @exception Exception
    *   - invalid stack symbol (id 1001)
    */
    void ConsistentStackSymbol(Idx idx) const;

    /**
    * Throw exception if vector of stack symbols contains stack symbols not
    * in generators stack symbol set 
    *
    * @exception Exception
    *   - invalid stack symbol (id 1001)
    */
    void ConsistentVectorStackSymbol(const std::vector<StackSymbol>& rVector) const;
    
   /**
    * Throw exception if vector of stack symbols contains stack symbols not
    * in generators stack symbol set 
    *
    * @exception Exception
    *   - invalid stack symbol (id 1001)
    */
    void ConsistentVectorStackSymbol(const std::vector<Idx>& rVector) const;
    
    /**
    * Throw exception if vector of stack symbols is empty 
    *
    * @exception Exception
    *   - invalid stack symbol (id 1001)
    */
    void EmptyVectorPopPush(const std::vector<StackSymbol>& rVector) const;
    
    /**
    * Throw exception if vector of stack symbols is empty 
    *
    * @exception Exception
    *   - invalid stack symbol (id 1001)
    */
    void EmptyVectorPopPush(const std::vector<Idx>& rVector) const;
	
    /**
     * Marks a state as being merged from other data type by setting mpMerge.
     * 
     * @param stateName
     *      the name of the state to mark
     * @param rMerge
     *      the merged state
     */
     void SetMerge(const std::string& stateName, MergeAbstract& rMerge);
     
     /**
      * Marks a state as being merged from other data type by setting mpMerge.
      * 
      * @param index
      *      the index of the state to mark
      * @param rMerge
      *      the merged state
      */
     void SetMerge(Idx index, MergeAbstract& rMerge);
     
     /**
      * Return the merge attribute of a state.
      * 
      * @param index
      *      the index of the state
      */
     const MergeAbstract* Merge(Idx index) const;
     
     /**
      * Marks a state as being derived from the intersection with a DFA.
      * 
      * @param stateName
      *      the name of the state to mark
      * @param dfaIndex
      *      the index of the DFA state
      */
     void SetDfaState(const std::string& stateName, Idx dfaIndex);
     
     /**
      * Marks a state as being derived from the intersection with a DPA.
      * 
      * @param index
      *      the index of the state to mark
      * @param dfaIndex
      *      the index of the DFA state
      */
     void SetDfaState(Idx index, Idx dfaIndex);
     
     /**
      * Return the dfaState attribute of a state.
      * 
      * @param index
      *      the index of the state
      */
     Idx DfaState(Idx index) const;
     
    /** 
     * Check if generator is valid 
     *
     * @return 
     *   Success
     */
    virtual bool Valid(void);
  
}; //end class TpdGenerator

/** Convenience typedef for std PushdownGenerator */
typedef TpdGenerator<AttributePushdownGlobal, AttributePushdownState, AttributeCFlags,AttributePushdownTransition>
  PushdownGenerator;
  
typedef TaTransSet<AttributePushdownTransition> PdTransSet;


// convenient scope macros  
#define THIS TpdGenerator<GlobalAttr, StateAttr, EventAttr, TransAttr>
#define BASE TcGenerator<GlobalAttr, StateAttr, EventAttr, TransAttr>
#define TEMP template <class GlobalAttr, class StateAttr, class EventAttr, class TransAttr>

// TpdGenerator(void)
TEMP THIS::TpdGenerator(void) : BASE() {
  // set basic members (cosmetic)
  BASE::pGlobalAttribute->mStackSymbols.Name("StackSymbols");
  BASE::pGlobalAttribute->mpStackSymbolTable=StackSymbolSet::StaticSymbolTablep();
  FD_DG("PushdownGenerator(" << this << ")::PushdownGenerator() with csymtab " 
    << (BASE::pGlobalAttribute->mpStackSymbolTable ));
}



// TpdGenerator(rOtherGen)
TEMP THIS::TpdGenerator(const TpdGenerator& rOtherGen) : BASE(rOtherGen) {
  FD_DG("PushdownGenerator(" << this << ")::PushdownGenerator(rOtherGen) with csymtab" 
    << "(BASE::pGlobalAttribute->mpStackSymbolTable)" );
}

// TpdGenerator(rOtherGen)
TEMP THIS::TpdGenerator(const vGenerator& rOtherGen) : BASE(rOtherGen) {
  // set basic members (cosmetic)
  BASE::pGlobalAttribute->mStackSymbols.Name("StackSymbols");
  BASE::pGlobalAttribute->mpStackSymbolTable=StackSymbolSet::StaticSymbolTablep();
  FD_DG("PushdownGenerator(" << this << ")::PushdownGenerator(rOtherGen) with csymtab" 
    << "(BASE::pGlobalAttribute->mpStackSymbolTable)" );
}

// TpdGenerator(rFilename)
TEMP THIS::TpdGenerator(const std::string& rFileName) : BASE(rFileName) {
  FD_DG("PushdownGenerator(" << this << ")::PushdownGenerator(" << rFileName << ") with csymtab" 
    << "(BASE::pGlobalAttribute->mpStackSymbolTable)" );
}

//GlobalStackSymbolTablep
TEMP SymbolTable* THIS::GlobalStackSymbolTablep(void) {
  return StackSymbolSet::StaticSymbolTablep();
}

// StackSymbolTablep()
TEMP SymbolTable* THIS::StackSymbolTablep(void) const {
  return BASE::pGlobalAttribute->mpStackSymbolTable;
}

// StackSymbolTablep(pSymTab)
TEMP void THIS::StackSymbolTablep(SymbolTable* pSymTab) {
   BASE::Clear(); // TODO: relax this
   BASE::pGlobalAttribute->mpStackSymbolTable=pSymTab;
}

// New
TEMP THIS* THIS::New(void) const {
  // allocate
  THIS* res = new THIS;
  // fix base data
  res->EventSymbolTablep(BASE::mpEventSymbolTable);
  res->mStateNamesEnabled=BASE::mStateNamesEnabled;
  res->mReindexOnWrite=BASE::mReindexOnWrite;  
  // fix my data
  res->StackSymbolTablep(StackSymbolTablep());
  return res;
}

// Copy
TEMP THIS* THIS::Copy(void) const {
  // allocate
  THIS* res = new THIS(*this);
  // done
  return res;
}

// NewTGen
TEMP THIS THIS::NewPdGen(void) const {
  // call base (fixes by assignment constructor)
  THIS res= BASE::NewCGen();
  // fix my data
  res.StackSymbolTablep(StackSymbolTablep());
  return res;
}

// StackSymbolsSize() const
TEMP Idx THIS::StackSymbolsSize(void) const {
  return BASE::pGlobalAttribute->mStackSymbols.Size();
}

// StackSymbols()
TEMP const StackSymbolSet& THIS::StackSymbols(void) const {
  return BASE::pGlobalAttribute->mStackSymbols;
}

// StackSymbolssp()
TEMP StackSymbolSet* THIS::StackSymbolsp(void) {
  return &BASE::pGlobalAttribute->mStackSymbols;
}

// InjectStackSymbols(set)
TEMP void THIS::InjectStackSymbols(const StackSymbolSet& newstacksymbols) {
  BASE::pGlobalAttribute->mStackSymbols=newstacksymbols;
  BASE::pGlobalAttribute->mStackSymbols.Name("StackSymbols");
}

// StackSymbolName(index)
TEMP std::string THIS::StackSymbolName(Idx index) const {
  return BASE::pGlobalAttribute->mStackSymbols.SymbolicName(index);
}

//StackSymbolObj(index)
TEMP StackSymbol THIS::StackSymbolObj(Idx index) const {
  return StackSymbol(BASE::pGlobalAttribute->mStackSymbols.SymbolicName(index));
}

// StackSymbolIndex(name)
TEMP Idx THIS::StackSymbolIndex(const std::string& rName) const {
  return BASE::pGlobalAttribute->mStackSymbols.Index(rName);
}

// InsStackSymbol(name)
TEMP Idx THIS::InsStackSymbol(const std::string& rName) {
  return BASE::pGlobalAttribute->mStackSymbols.Insert(rName);
}

//InsStackSymbol(symbol)
TEMP Idx THIS::InsStackSymbol(const StackSymbol& rSymbol){
  return BASE::pGlobalAttribute->mStackSymbols.Insert(rSymbol.Symbol());
}

// InsStackSymbols(set)
TEMP void THIS::InsStackSymbols(const StackSymbolSet& rStackSymbolSet) {
  BASE::pGlobalAttribute->mStackSymbols.InsertSet(rStackSymbolSet);
}

// SetStackBottom(name)
TEMP Idx THIS::SetStackBottom(const std::string& rName) {
  Idx i = BASE::pGlobalAttribute->mStackSymbols.Insert(rName);
  BASE::pGlobalAttribute->mStackBottom = i;
  return i;
}

//SetStackBottom(symbol)
TEMP Idx THIS::SetStackBottom(const StackSymbol& rSymbol){
  Idx i = BASE::pGlobalAttribute->mStackSymbols.Insert(rSymbol.Symbol());
  BASE::pGlobalAttribute->mStackBottom = i;
  return i;
}

//SetStackBottom(index)
TEMP Idx THIS::SetStackBottom(const Idx idx){
  if(!ExistsStackSymbol(idx)) {
      std::stringstream errstr;
      errstr << "stack symbol with index " << idx << " not found in generator. " << std::endl;
      throw Exception("PushdownGenerator::SetStackBottom(idx)", errstr.str(), 200);
  }
  BASE::pGlobalAttribute->mStackBottom = StackSymbol(StackSymbolName(idx));
  return idx;
}

//StackBottom()
TEMP Idx THIS::StackBottom() const{
  return BASE::pGlobalAttribute->mStackBottom;
}

// DelStackSymbol(index)
TEMP bool THIS::DelStackSymbol(Idx index) {
  FD_DG("PushdownGenerator(" << this << ")::DelStackSymbol(" << index << ")");
  return BASE::pGlobalAttribute->mStackSymbols.Erase(index);
}

// DelStackSymbol(name)
TEMP bool THIS::DelStackSymbol(const std::string& rName) {
  Idx index=BASE::pGlobalAttribute->mStackSymbols.Index(rName);
  return DelStackSymbol(index);
}

// DelStackSymbols(set)
TEMP void THIS::DelStackSymbols(const StackSymbolSet& rStackSymbols) {
  StackSymbolSet::Iterator it=StackSymbolsBegin();
  while(it!=StackSymbolsEnd()){
    DelStackSymbol(*(it++)); // fixed: 2013-12-17
  }
}

// ExistsStackSymbol(index)
TEMP bool THIS::ExistsStackSymbol(Idx index) const {
  return BASE::pGlobalAttribute->mStackSymbols.Exists(index);
}

// ExistsStackSymbol(name)
TEMP bool THIS::ExistsStackSymbol(
  const std::string& rName) const {
  return BASE::pGlobalAttribute->mStackSymbols.Exists(rName);
}

// FindStackSymbol(index)
TEMP StackSymbolSet::Iterator THIS::FindStackSymbol(Idx index) const {
  return BASE::pGlobalAttribute->mStackSymbols.Find(index);
}

// FindStackSymbol(name)
TEMP StackSymbolSet::Iterator THIS::FindStackSymbol(const std::string& rName) const {
  return BASE::pGlobalAttribute->mStackSymbols.Find(rName);
}

// iterator StackSymbolsBegin() const
TEMP StackSymbolSet::Iterator THIS::StackSymbolsBegin(void) const {
  return BASE::pGlobalAttribute->mStackSymbols.Begin();
}

// iterator StackSymbolsEnd() const
TEMP StackSymbolSet::Iterator THIS::StackSymbolsEnd(void) const {
  return BASE::pGlobalAttribute->mStackSymbols.End();
}
//StackSymbolsToIndices
TEMP std::vector<Idx> THIS::StackSymbolsToIndices(const std::vector<StackSymbol> symbols) const{
  std::vector<StackSymbol>::const_iterator ssit;
  std::vector<Idx> rV;
  for(ssit = symbols.begin(); ssit != symbols.end(); ssit++){
    rV.push_back(StackSymbolIndex(ssit->Symbol()));
  }
  return rV;
}

TEMP Idx THIS::TransRelSize() const{
  
  TransSet::Iterator transit;
  Idx s = 0;
  for(transit = BASE::TransRelBegin(); transit != BASE::TransRelEnd(); transit++){
    s += PopPush(*transit).size();
  }
  return s;
}

// SetTransition(rX1, rEv, rX2)
TEMP bool THIS::SetTransition(const std::string& rX1, const std::string& rEv, const std::string& rX2) {
  return BASE::SetTransition(rX1,rEv,rX2);
}


// SetTransition(x1, ev, x2)
TEMP bool THIS::SetTransition(Idx x1, Idx ev, Idx x2) {
  return BASE::SetTransition(Transition(x1,ev,x2));
}

// SetTransition(rTransition, rAttr)
TEMP bool THIS::SetTransition(const Transition& rTransition, const TransAttr& rAttr) {
  return BASE::SetTransition(rTransition,rAttr);
}

// SetTransition(trans,....)

TEMP bool THIS::SetTransition(const Transition& rTrans,
    const std::vector<StackSymbol>& rPop, const std::vector<StackSymbol>& rPush) {
//   FD_DG("PushdownGenerator(" << this << ")::SetTransition(" << (BASE::TStr(rTrans)) <<", " <<
//     ", PopVector" << ", " << "PushVector" << ") const");
// #ifdef FAUDES_CHECKED
//   EmptyVectorPopPush(rPop);
//   EmptyVectorPopPush(rPush);
// #endif
//   BASE::SetTransition(rTrans);
//   //get the transition attribute or take new one if it does not exist
//   TransAttr attr;
//   if(BASE::TransAttributep(rTrans) != 0){
//     attr = *BASE::TransAttributep(rTrans);
//   }
//   std::pair<std::vector<Idx>, std::vector<Idx> > popPushPair;
//   popPushPair.first = StackSymbolsToIndices(rPop);
//   popPushPair.second = StackSymbolsToIndices(rPush);
//   //add new PopPushPair
//   attr.mPopPush.insert(popPushPair);
// #ifdef FAUDES_CHECKED
//   ConsistentVectorStackSymbol(rPop);
//   ConsistentVectorStackSymbol(rPush);
// #endif
//   return BASE::SetTransition(rTrans,attr);
  return SetTransition(rTrans, StackSymbolsToIndices(rPop), StackSymbolsToIndices(rPush));
}

TEMP bool THIS::SetTransition(const Transition& rTrans,
    const std::vector<Idx>& rPop, const std::vector<Idx>& rPush) {
  FD_DG("PushdownGenerator(" << this << ")::SetTransition(" << (BASE::TStr(rTrans)) <<", " <<
    ", PopVector" << ", " << "PushVector" << ") const");
#ifdef FAUDES_CHECKED
  EmptyVectorPopPush(rPop);
  EmptyVectorPopPush(rPush);
#endif
  BASE::SetTransition(rTrans);
  //get the transition attribute or take new one if it does not exist
  TransAttr attr;
  if(BASE::TransAttributep(rTrans) != 0){
    attr = *BASE::TransAttributep(rTrans);
  }
  std::pair<std::vector<Idx>, std::vector<Idx> > popPushPair;
  popPushPair.first = rPop;
  popPushPair.second = rPush;
  //add new PopPushPair
  attr.mPopPush.insert(popPushPair);
#ifdef FAUDES_CHECKED
  ConsistentVectorStackSymbol(rPop);
  ConsistentVectorStackSymbol(rPush);
#endif
  return BASE::SetTransition(rTrans,attr);
}

// SetTransition(x1,ev,x2, ...)
TEMP bool THIS::SetTransition(Idx x1, Idx ev, Idx x2,
    const std::vector<StackSymbol>& rPop, const std::vector<StackSymbol>& rPush) {
  return SetTransition(Transition(x1,ev,x2),rPop,rPush);
}

// SetTransition(x1,ev,x2, ...)
TEMP bool THIS::SetTransition(Idx x1, Idx ev, Idx x2,
    const std::vector<Idx>& rPop, const std::vector<Idx>& rPush) {
  return SetTransition(Transition(x1,ev,x2),rPop,rPush);
}

// SetTransition(X1,Ev,X2, ...)
TEMP bool THIS::SetTransition(
    const std::string& rX1, const std::string& rEv, const std::string& rX2,
    const std::vector<StackSymbol>& rPop, const std::vector<StackSymbol>& rPush) {
    FD_DG("PushdownGenerator(" << this << ")::SetTransition(" << rX1 << " " << rEv <<" " << rX2 <<
      ", PopVector" << ", " << "PushVector" << ") const");
  //try to add transition, will do nothing if transition exists
  bool res=BASE::SetTransition(rX1,rEv,rX2);
  //get transition via iterator
  Transition rTrans = *(BASE::FindTransition(rX1,rEv,rX2));
#ifdef FAUDES_CHECKED
  EmptyVectorPopPush(rPop);
  EmptyVectorPopPush(rPush);
#endif
  //get the transition attribute or take new one if it does not exist
  TransAttr attr;
  if(BASE::TransAttributep(rTrans) != 0){
    attr = *BASE::TransAttributep(rTrans);
  }
  std::pair<std::vector<Idx>, std::vector<Idx> > popPushPair;
  popPushPair.first = StackSymbolsToIndices(rPop);
  popPushPair.second = StackSymbolsToIndices(rPush);
  //add new PopPushPair
  attr.mPopPush.insert(popPushPair);
#ifdef FAUDES_CHECKED
  ConsistentVectorStackSymbol(rPop);
  ConsistentVectorStackSymbol(rPush);
#endif
  BASE::TransAttribute(Transition(BASE::StateIndex(rX1),BASE::EventIndex(rEv),BASE::StateIndex(rX2)),attr);
  return res;
}

// SetTransition(X1,Ev,X2, ...)
TEMP bool THIS::SetTransition(
    const std::string& rX1, const std::string& rEv, const std::string& rX2,
    const std::vector<Idx>& rPop, const std::vector<Idx>& rPush) {
  
  Transition(BASE::StateIndex(rX1),BASE::EventIndex(rEv),BASE::StateIndex(rX2));
  return SetTransition(Transition(BASE::StateIndex(rX1),BASE::EventIndex(rEv),BASE::StateIndex(rX2)),rPop,rPush);
}

//clearTransition(x1,ev,x2,rPop,rPush)
TEMP bool THIS::ClrTransition(Idx x1, Idx ev, Idx x2,
    const std::vector<StackSymbol>& rPop, const std::vector<StackSymbol>& rPush){
  return ClrTransition(Transition(x1,ev,x2),rPop,rPush);
}

//clearTransition(x1,ev,x2,rPop,rPush)
TEMP bool THIS::ClrTransition(Idx x1, Idx ev, Idx x2,
    const std::vector<Idx>& rPop, const std::vector<Idx>& rPush){
  return ClrTransition(Transition(x1,ev,x2),rPop,rPush);
}

//clearTransition(rTrans,rPop,rPush)
TEMP bool THIS::ClrTransition(const Transition& rTrans, const std::vector<StackSymbol>& rPop,const std::vector<StackSymbol>& rPush){
  
//  //check for existence of base transition
//   if(!BASE::ExistsTransition(rTrans)){
//     return false;
//   }
//   
//   //delete pop/push pair in said transition
//   if(!BASE::pTransRel->Attributep(rTrans)->ClrPopPush(StackSymbolsToIndices(rPop),StackSymbolsToIndices(rPush))){
//     return false;
//   }
//   
//   //delete transition if popPush is empty
//   if(PopPush(rTrans).empty()){
//     BASE::ClrTransition(rTrans);
//   }
//   return true;
  return ClrTransition(rTrans,StackSymbolsToIndices(rPop),StackSymbolsToIndices(rPush));
}

//clearTransition(rTrans,rPop,rPush)
TEMP bool THIS::ClrTransition(const Transition& rTrans, const std::vector<Idx>& rPop,const std::vector<Idx>& rPush){
  
  //check for existence of base transition
  if(!BASE::ExistsTransition(rTrans)){
    return false;
  }
  
  //delete pop/push pair in said transition
  if(!BASE::pTransRel->Attributep(rTrans)->ClrPopPush(rPop, rPush)){
    return false;
  }
  
  //delete transition if popPush is empty
  if(PopPush(rTrans).empty()){
    BASE::ClrTransition(rTrans);
  }
  return true;
}

//PopPush(trans)
TEMP const PopPushSet& THIS::PopPush(const Transition& rTrans) const{
#ifdef FAUDES_CHECKED
  if(!BASE::ExistsTransition(rTrans)) {
      std::stringstream errstr;
      errstr << "transition " << BASE::TStr(rTrans) << " not found " << std::endl;
      throw Exception("PushdownGenerator::PopPush(trans)", errstr.str(), 200);
  }
#endif
  return BASE::pTransRel->Attribute(rTrans).PopPush();
}

//PopPushBegin(trans)
TEMP PopPushSet::const_iterator THIS::PopPushBegin(const Transition& rTrans) const {
  return BASE::pTransRel->Attribute(rTrans).PopPush().begin();
}

//PopPushEnd(trans)
TEMP PopPushSet::const_iterator THIS::PopPushEnd(const Transition& rTrans) const {
  return BASE::pTransRel->Attribute(rTrans).PopPush().end();
}

//IsStackSymbolLambda(index)
TEMP bool THIS::IsStackSymbolLambda(Idx index) const{
  if(StackSymbolName(index).compare(FAUDES_PD_LAMBDA) == 0)
    return true;
  else
    return false;
}

//IsEventLambda(index)
TEMP bool THIS::IsEventLambda(Idx index) const{
  if(BASE::EventName(index).compare(FAUDES_PD_LAMBDA) == 0)
    return true;
  else
    return false;
}


// ConsistentStackSymbol(rStackSymbol)
TEMP void THIS::ConsistentStackSymbol(const StackSymbol& rStackSymbol) const {
  FD_DG("PushdownGenerator(" << this << ")::ConsistentStackSymbol(rStackSymbol)");
  if(!StackSymbols().Exists(rStackSymbol.Symbol())) {
    std::stringstream errstr;
    errstr << "stack symbol table mismatch (symbol " << rStackSymbol.mSymbol << " does not exist)" << std::endl;
    throw Exception("PushdownGenerator::ConsistentStackSymbol", errstr.str(), 1001);
  }
  FD_DG("PushdownGenerator(" << this << ")::ConsistentStackSymbol(rStackSymbol): ok");
}

// ConsistentStackSymbol(Idx)
TEMP void THIS::ConsistentStackSymbol(Idx idx) const {
  FD_DG("PushdownGenerator(" << this << ")::ConsistentStackSymbol(idx)");
  if(!StackSymbols().Exists(idx)) {
    std::stringstream errstr;
    errstr << "stack symbol table mismatch (symbol with idx " << idx << " does not exist)" << std::endl;
    throw Exception("PushdownGenerator::ConsistentStackSymbol", errstr.str(), 1001);
  }
  FD_DG("PushdownGenerator(" << this << ")::ConsistentStackSymbol(idx): ok");
}

// ConsistentVectorStackSymbol(vector)
TEMP void THIS::ConsistentVectorStackSymbol(const std::vector<StackSymbol>& rVector) const {
  FD_DG("PushdownGenerator(" << this << ")::ConsistentVectorStackSymbol(rVector)");
  std::vector<StackSymbol>::const_iterator it;
  it = rVector.begin();
  for ( ; it < rVector.end(); it++){
    if(!StackSymbols().Exists(it->Symbol())) {
      std::stringstream errstr;
      errstr << "stack symbol table mismatch (symbol " << it->mSymbol << " does not exist)" <<std::endl;
      throw Exception("PushdownGenerator::ConsistentVectorStackSymbol", errstr.str(), 1001);
    }
  }
  FD_DG("PushdownGenerator(" << this << ")::ConsistentVectorStackSymbol(rVector): ok");
}

// ConsistentVectorStackSymbol(vector)
TEMP void THIS::ConsistentVectorStackSymbol(const std::vector<Idx>& rVector) const {
  FD_DG("PushdownGenerator(" << this << ")::ConsistentVectorStackSymbol(rVector)");
  std::vector<Idx>::const_iterator it;
  it = rVector.begin();
  for ( ; it < rVector.end(); it++){
    if(!StackSymbols().Exists(*it)) {
      std::stringstream errstr;
      errstr << "stack symbol table mismatch (symbol with index " << *it << " does not exist)" <<std::endl;
      throw Exception("PushdownGenerator::ConsistentVectorStackSymbol", errstr.str(), 1001);
    }
  }
  FD_DG("PushdownGenerator(" << this << ")::ConsistentVectorStackSymbol(rVector): ok");
}

// EmptyVector(vector)
TEMP void THIS::EmptyVectorPopPush(const std::vector<StackSymbol>& rVector) const {
  FD_DG("PushdownGenerator(" << this << ")::EmptyVector(rVector)");
  if(rVector.empty()) {
    std::stringstream errstr;
    errstr << "empty vector not allowed in pop or push" <<std::endl;
    throw Exception("PushdownGenerator::EmptyVector", errstr.str(), 1001);
  }
  FD_DG("PushdownGenerator(" << this << ")::EmptyVectorPopPush(rVector): ok");
}

// EmptyVector(vector)
TEMP void THIS::EmptyVectorPopPush(const std::vector<Idx>& rVector) const {
  FD_DG("PushdownGenerator(" << this << ")::EmptyVector(rVector)");
  if(rVector.empty()) {
    std::stringstream errstr;
    errstr << "empty vector not allowed in pop or push" <<std::endl;
    throw Exception("PushdownGenerator::EmptyVector", errstr.str(), 1001);
  }
  FD_DG("PushdownGenerator(" << this << ")::EmptyVectorPopPush(rVector): ok");
}


//SetMerge(stateName,rMerge)
TEMP void THIS::SetMerge(const std::string& stateName, MergeAbstract& rMerge){
  Idx index=BASE::StateIndex(stateName);
  BASE::pStates->Attributep(index)->SetMerge(rMerge);
}

//SetMerge(index,rMerge)
TEMP void THIS::SetMerge(Idx index, MergeAbstract& rMerge){
  BASE::pStates->Attributep(index)->SetMerge(rMerge);
}

//Merge(index)
TEMP const MergeAbstract* THIS::Merge(Idx index) const{
  return BASE::pStates->Attributep(index)->Merge();
}

//SetDfaState(stateName, dpaIndex)
TEMP void THIS::SetDfaState(const std::string& stateName, Idx dfaIndex){
  Idx index=BASE::StateIndex(stateName);
  BASE::pStates->Attributep(index)->DfaState(dfaIndex);
}

//SetDfaState(index, dpaIndex)
TEMP void THIS::SetDfaState(Idx index, Idx dfaIndex){
  BASE::pStates->Attributep(index)->DfaState(dfaIndex);
}

//DfaState(index)
TEMP Idx THIS::DfaState(Idx index) const{
  return BASE::pStates->Attributep(index)->DfaState();
}

// Valid() TODO checks?
TEMP bool THIS::Valid(void) {
    FD_DV("PushdownGenerator(" << this << ")::Valid()");
    //call base
    if(!BASE::Valid()) return false;
    // check my names
   
    // check my clockset
    
    // check all clocksymboltables
    
    return true;
}
  

// clean up defs (tmoor)
#undef THIS 
#undef BASE 
#undef TEMP 


  
} // namespace faudes


#endif

