/** @file pd_alg_lrp.cpp  functions related to parsers*/


/* Pushdown plugin for FAU Discrete Event Systems Library (libfaudes)

   Copyright (C) 2013  Stefan Jacobi, Sven Schneider, Anne-Kathrin Hess

*/

#include "pd_alg_lrp.h"

namespace faudes {

  
/* *************************
 * MachineGoto
 * *************************/
std::set<Idx> GeneratorGoto(const GotoGenerator& gotoGen, Idx state, const GrammarSymbolPtr& symbol){
  
  //state set to return
  std::set<Idx> successorStates;
  
  TransSet::Iterator transit;
  //iterate over all states with state as a start state
  for(transit = gotoGen.TransRelBegin(state); transit != gotoGen.TransRelEnd(state); transit++){
    
    //if the symbol is the same, insert end state of the transition as successor state
    if(*gotoGen.Symbol(*transit) == *symbol){
      successorStates.insert(transit->X2);
    }
  }
  return successorStates;
}
  
  
/* *************************
 * GeneratorGotoSeq
 * *************************/
std::vector<Idx> GeneratorGotoSeq(const GotoGenerator& gotoGen, Idx startState, const GrammarSymbolVector& word){
  
  //state sequence to return
  std::vector<Idx> rStateSequence;
  
  //for later use
  Terminal* lambda = new Terminal(PushdownGenerator::GlobalEventSymbolTablep()->Index(FAUDES_PD_LAMBDA));
  GrammarSymbolPtr lambdaPtr(lambda);
  

  //if the word is not empty (or does not contain lambda), it is of the form w = a.w'
  if(!word.empty()){
    if(*word.front() != *lambda){
      
      //get front symbol a and remaining word w'
      GrammarSymbolPtr frontSymbol = word.front();
      GrammarSymbolVector remainingWord(word.begin() + 1, word.end());
      
      //get next state in the sequence
      std::set<Idx> nextStateSet = GeneratorGoto(gotoGen, startState, frontSymbol);
      Idx nextState;
      if(!nextStateSet.empty()){
        nextState = *nextStateSet.begin();
        
        //append the next state in the sequence
        rStateSequence.push_back(nextState);
      }
      
      //get the rest of the sequence
      std::vector<Idx> remainingSequence = GeneratorGotoSeq(gotoGen, nextState, remainingWord);
      
      //append the rest of the sequence
      rStateSequence.insert(rStateSequence.end(), remainingSequence.begin(), remainingSequence.end());
    } 
  }
  return rStateSequence;
}

/* *************************
 * LrpShiftRules
 * *************************/
std::set<Lr1ParserAction> LrpShiftRules(const Grammar& gr, const Grammar& augGr, const GotoGenerator& gotoGen, uint k){
  
  //parser actions to return
  std::set<Lr1ParserAction> rActions;
  
  //lambda terminal for later use
  Terminal* lambda = new Terminal(PushdownGenerator::GlobalEventSymbolTablep()->Index(FAUDES_PD_LAMBDA));
  GrammarSymbolPtr lambdaPtr(lambda);
  
  StateSet::Iterator stateit;
  //iterate over all states q of the gotoGenerator
  for(stateit = gotoGen.StatesBegin(); stateit != gotoGen.StatesEnd(); stateit++){
    
    std::set<Terminal>::const_iterator tit;
    //iterate over all terminals a of the grammar
    for(tit = gr.TerminalsBegin(); tit != gr.TerminalsEnd(); tit++){
      
      //do not care for lambda
      if(tit->IsLambda()){
        continue;
      }
      
      //convert terminal into GrammarSymbolPtr
      Terminal* a = new Terminal(*tit);
      GrammarSymbolPtr aPtr(a);
      
      //get the successor state q' for this terminal
      std::set<Idx> succStateSet = GeneratorGoto(gotoGen, *stateit, aPtr);
      
      //continue, if there is no successor state (there can't be too many since gotoGen 
      //is supposed to be deterministic)
      if(succStateSet.size() != 1){
        continue;
      }
      
      //extract the successor state
      Idx succState = *succStateSet.begin();
      
      std::set<Lr1Configuration>::const_iterator configit;
      //iterate over all configurations contained in the successor state
      for(configit = gotoGen.ConfigSet(*stateit).begin(); configit != gotoGen.ConfigSet(*stateit).end(); configit++){
        
        //only consider this configuration (A -> w1 . a w2, z) if the
        //production (A -> w1 a w2) is in the grammar.
        
        GrammarSymbolVector beforeDot, afterDot;
        //construct right-hand side of the production, but need to make sure to delete
        //unnecessary lambdas
        beforeDot = configit->BeforeDot();
        if(**beforeDot.begin() == *lambda){
          beforeDot.clear();
        }
        afterDot = configit->AfterDot();
        if(**afterDot.begin() == *lambda){
          afterDot.clear();
        }
        
        //create w1 a w2 from afterDot and beforeDot
        GrammarSymbolVector rhs(beforeDot);
        rhs.insert(rhs.end(), afterDot.begin(), afterDot.end());
        if(rhs.empty()){
          rhs.push_back(lambdaPtr);
        }
        
        //construct production
        GrammarProduction gp(configit->Lhs(), rhs);
        
        //test if the configuration is not in the grammar and if so, continue
        if(gr.GrammarProductions().find(gp) == gr.GrammarProductionsEnd()){
          continue;
        }
        
        //construct the word w2 z
        //w2 can't be lambda
        GrammarSymbolVector w2z(configit->AfterDot().begin() + 1, configit->AfterDot().end());
        
        //append z if it's not lambda
        if(!configit->Lookahead().IsLambda()){
          Terminal* lookahead = new Terminal(configit->Lookahead());
          GrammarSymbolPtr lookaheadPtr(lookahead);
          w2z.push_back(lookaheadPtr);
        }
        
        //get first terminals after w2z (will be either nothing or lambda)
        std::set<Terminal> firstTerminals = FirstLeq1(augGr, k - 1, w2z);
        
        std::set<Terminal>::const_iterator firsttit;
        //iterate over first terminals and add a parser action for each terminal y
        for(firsttit = firstTerminals.begin(); firsttit != firstTerminals.end(); firsttit++){
          
          //just to make sure that the terminal must be lambda
          if(!firsttit->IsLambda()){
            std::stringstream errstr;
            errstr << "While executing LrpRules(): Constructing shift actions and the terminal " << firsttit->Str() << ", which was expected to be lambda" << std::endl;
            throw Exception("LrpRules", errstr.str(), 1001);
          }
          
          //construct left-hand side of the rule (q | a y) (which is (q | a),
          //because y is lambda)
          std::vector<Idx> stateStack;
          stateStack.push_back(*stateit);
          Lr1ParserActionElement actionLhs(stateStack, *a);
          
          //construct right-hand side of the rule (q q' | y)
          stateStack.push_back(succState);
          Lr1ParserActionElement actionRhs(stateStack, *lambda);
          
          //construct shift action (q | a) -> (q q' | lambda) and insert into 
          //set of actions
          Lr1ParserAction action(actionLhs, actionRhs);
          rActions.insert(action);
        }
      }
    }
  }
  return rActions;
}

/* *************************
 * LrpReduceRules
 * *************************/
std::set<Lr1ParserAction> LrpReduceRules(const Grammar& gr, const Grammar& augGr, const GotoGenerator& gotoGen, uint k){
  
  //parser actions to return
  std::set<Lr1ParserAction> rActions;
  
  //lambda terminal for later use
  Terminal* lambda = new Terminal(PushdownGenerator::GlobalEventSymbolTablep()->Index(FAUDES_PD_LAMBDA));
  
  StateSet::Iterator stateit;
  //iterate over all states q of the gotoGenerator
  for(stateit = gotoGen.StatesBegin(); stateit != gotoGen.StatesEnd(); stateit++){
    
    std::set<Lr1Configuration>::const_iterator configit;
    //iterate over all configurations  contained in the state q
    for(configit = gotoGen.ConfigSet(*stateit).begin(); configit != gotoGen.ConfigSet(*stateit).end(); configit++){
      
      //only consider this configuration if it is of the form (A -> lambda . w, z),
      //i. e. if beforeDot is lambda      
      if(**configit->BeforeDot().begin() != *lambda){
        continue;
      }
      
      //only consider this configuration if the production (A -> w) is in the grammar
      GrammarProduction gp(configit->Lhs(), configit->AfterDot());
      if(gr.GrammarProductions().find(gp) == gr.GrammarProductionsEnd()){
        continue;
      }
      
      //convert A into GrammarSymbolPtr and get successor state qa of q with the symbol A
      Nonterminal* a = new Nonterminal(configit->Lhs());
      GrammarSymbolPtr aPtr(a);
      std::set<Idx> succStateSet = GeneratorGoto(gotoGen, *stateit, aPtr);
      
      //if there is no successor state, continue
      if(succStateSet.size() != 1){
        continue;
      }
      Idx succState = *succStateSet.begin();
      
      //get successor state sequence qs of q with the word w
      std::vector<Idx> succStateSequence = GeneratorGotoSeq(gotoGen, *stateit, configit->AfterDot());
      
      //join state q and state sequence qs
      std::vector<Idx> qQs;
      qQs.push_back(*stateit);
      qQs.insert(qQs.end(), succStateSequence.begin(), succStateSequence.end());
      
      //get last state of qQs state sequence
      Idx lastState = qQs.back();
      
      std::set<Lr1Configuration>::const_iterator lsconfigit;
      //iterate over configurations in the last state
      for(lsconfigit = gotoGen.ConfigSet(lastState).begin(); lsconfigit != gotoGen.ConfigSet(lastState).end(); lsconfigit++){
        
        //only consider this configuration if it is of the form (A -> w . lambda, y),
        if(lsconfigit->Lhs() != configit->Lhs()){
          continue;
        }
        if(!EqualsGsVector(lsconfigit->BeforeDot(), configit->AfterDot())){
          continue;
        }
        if(**lsconfigit->AfterDot().begin() != *lambda){
          continue;
        }
        
        //get the lookahead terminal
        Terminal lookahead(lsconfigit->Lookahead());
        
        //add parser action for each configuration
        //construct left-hand side (q qs | y)
        Lr1ParserActionElement lhs(qQs, lookahead);
        
        //construct right-hand side (q qa | y)
        std::vector<Idx> stateStack;
        stateStack.push_back(*stateit);
        stateStack.push_back(succState);
        Lr1ParserActionElement rhs(stateStack, lookahead);
        
        //construct reduce action (q qs | y) -> (q qa | y) and insert it into set
        //of actions
        Lr1ParserAction action(lhs, rhs, gp);
        rActions.insert(action);
      }
    }
  }
  return rActions;
}

/* *************************
 * LrpRules
 * *************************/
std::set<Lr1ParserAction> LrpRules(const Grammar& gr, const Grammar& augGr, const GotoGenerator& gotoGen, uint k){
  
  //parser actions to return
  std::set<Lr1ParserAction> rActions;
  
  //get shift actions
  rActions = LrpShiftRules(gr, augGr, gotoGen, k);
  
  //get reduce actions
  std::set<Lr1ParserAction> reduceActions = LrpReduceRules(gr, augGr, gotoGen, k);
  rActions.insert(reduceActions.begin(), reduceActions.end());

  return rActions;
}

/* *************************
 * Lrp
 * *************************/
Lr1Parser Lrp(const Grammar& gr, const Grammar& augGr, const GotoGenerator& gotoGen, uint k, const Terminal& augSymbol){
  
  //parser to return
  Lr1Parser rParser;
  
  //convert augSymbol $ into GrammarSymbolPtr
  Terminal* augSymbolTemp = new Terminal(augSymbol);
  GrammarSymbolPtr augSymbolPtr(augSymbolTemp);
  
  //convert the grammar's start symbol q0 into GrammarSymbolPtr
  Nonterminal* q0 = new Nonterminal(gr.StartSymbol());
  GrammarSymbolPtr q0Ptr(q0);
  
  //get successor state of the initial goto generator state and $
  std::set<Idx> succStateSetAug = GeneratorGoto(gotoGen, gotoGen.InitState(), augSymbolPtr);
  
  //get successor state sequence of the initial goto generator state and $ q0
  GrammarSymbolVector word;
  word.push_back(augSymbolPtr);
  word.push_back(q0Ptr);
  std::vector<Idx> succStateSeqAugQ0 = GeneratorGotoSeq(gotoGen, gotoGen.InitState(), word);
  
  //get successor state sequence of the initial goto generator state and $ q0 $
  word.push_back(augSymbolPtr);
  std::vector<Idx> succStateSeqAugQ0Aug = GeneratorGotoSeq(gotoGen, gotoGen.InitState(), word);
  
  //if the successor states for $, $ q0 and $ q0 $ exist (i. e. they are not empty)
  if(!succStateSetAug.empty() && !succStateSeqAugQ0.empty() && !succStateSeqAugQ0Aug.empty()){
    
    //get the successor state for $
    Idx succStateAug = *succStateSetAug.begin();
    
    //get the last state of the $ q0 sequence
    Idx lastStateAugQ0 = succStateSeqAugQ0.back();
    
    //get the last state of the $ q0 $ sequence
    Idx lastStateAugQ0Aug = succStateSeqAugQ0Aug.back();
    
    StateSet::Iterator stateit;
    //insert all states of the gotoGen as nonterminals of the parser, but exclude
    //the start state and the end state
    for(stateit = gotoGen.StatesBegin(); stateit != gotoGen.StatesEnd(); stateit++){
      if(*stateit != gotoGen.InitState() && *stateit != lastStateAugQ0Aug){
        rParser.InsNonterminal(*stateit);
      }
    }
    
    std::set<Terminal>::const_iterator tit;
    //insert all the augmented grammar's terminals as the parser's terminals
    for(tit = augGr.TerminalsBegin(); tit != augGr.TerminalsEnd(); tit++){
      rParser.InsTerminal(*tit);
    }
    
    //set the parser's start state
    rParser.SetStartState(succStateAug);
    
    //set the parser's final state
    rParser.SetFinalState(lastStateAugQ0);
    
    //get the parsing actions
    std::set<Lr1ParserAction> actions = LrpRules(gr, augGr, gotoGen, k);
    
    std::set<Lr1ParserAction>::const_iterator actit;
    //insert the parsing actions into the parser
    for(actit = actions.begin(); actit != actions.end(); actit++){
      rParser.InsAction(*actit);
    }
    
    //save $ for later reference in the parser
    rParser.SetAugSymbol(augSymbol);
  }
  
  return rParser;
}

/* *************************
 * Gp2Pp
 * *************************/
PushdownGenerator Gp2Pp(const GotoGenerator& gotoGen, const Lr1Parser& parser){
  
  //generator to return
  PushdownGenerator rPd;
  
  std::set<Idx>::const_iterator ntit;
  std::set<Terminal>::const_iterator tit;
  std::map<std::pair<Idx,Idx>, Idx> stateMap;
  std::map<std::pair<Idx,Idx>, Idx>::const_iterator statemapit;
  
  //iterate over the parser's nonterminals
  for(ntit = parser.Nonterminals().begin(); ntit != parser.Nonterminals().end();ntit++){
    
    //insert states for each pair of nonterminal and terminal of the parser
    for(tit = parser.Terminals().begin(); tit != parser.Terminals().end(); tit++){
        
      //insert new states. start/final states can be inserted if the current terminal
      //is lambda
      Idx newState;
      //start state
      if(parser.StartState() == *ntit && tit->IsLambda()){
        newState = rPd.InsInitState();
      }
      //final state
      else if(parser.FinalState() == *ntit && tit->IsLambda()){
        newState = rPd.InsMarkedState();
      }
      //normal state
      else{
        newState = rPd.InsState();
      }
      
      //set MergeStateEvent of that new state
      MergeStateEvent mse(*ntit, tit->Event());
      rPd.SetMerge(newState, mse);
      
      //save information on how the new states were created for later reference
      std::pair<Idx,Idx> key = std::make_pair(*ntit, tit->Event());
      stateMap.insert(std::make_pair(key, newState));
    }
    
    //the parser's nonterminals are the generator's stack symbols (needs conversion to
    //string)
    std::stringstream s;
    s << *ntit;
    rPd.InsStackSymbol(s.str());
  }
  
  //lambda is always a stack symbol
  rPd.InsStackSymbol(FAUDES_PD_LAMBDA);
  
  //add "0" as stack bottom
  rPd.SetStackBottom("0");
  
  //the parser's terminals are the generator's events
  for(tit = parser.Terminals().begin(); tit != parser.Terminals().end(); tit++){
    
    //insert event
    rPd.InsEvent(tit->Str());
  }
  
  //insert final state of the pushdown generator. the final state is a tuple of
  //the goto generator's final state and lambda. first check if that state already
  //has been added
  std::pair<Idx, Idx> finalState = std::make_pair(*gotoGen.MarkedStatesBegin(), rPd.EventIndex(FAUDES_PD_LAMBDA));
  statemapit = stateMap.find(finalState);
  
  std::set<Lr1ParserAction>::const_iterator actit;
  //iterate over parser actions to create transitions
  for(actit = parser.Actions().begin(); actit != parser.Actions().end(); actit++){
    
    //get left-hand side and right-hand side of the action
    Lr1ParserActionElement lhs = actit->Lhs();
    Lr1ParserActionElement rhs = actit->Rhs();
    
    //if the actions left-hand side next terminal is lambda, it is an unexpected rule,
    //so throw an error
    if(lhs.NextTerminal().IsLambda()){
      std::stringstream errstr;
      errstr << "While executing Gp2Pp(): Tried to create transition from parser action " << actit->Str() << ", but no rule exists for converting transitions with lambda in the left-hand side's next terminal. " << std::endl;
      throw Exception("Gp2Pp", errstr.str(), 1001);
    }
    
    //for each shift action (A | a) -> (AB | lambda)
    else if(lhs.StateStack().size() == 1 && 
      rhs.StateStack().size() == 2 && 
      rhs.NextTerminal().IsLambda() &&
      lhs.StateStack().front() == rhs.StateStack().front())
    {
      //extract A, B, a and lambda
      Idx stateA = rhs.StateStack().front();
      Idx stateB = rhs.StateStack().back();
      Idx eventA = lhs.NextTerminal().Event();
      Idx eventLambda = rhs.NextTerminal().Event();
      
      //get state (A, lambda)
      statemapit = stateMap.find(std::make_pair(stateA,eventLambda));
      if(statemapit == stateMap.end()){
        std::stringstream errstr;
        errstr << "While executing Gp2Pp(): Tried to find a pushdown generator state for the pair of goto generator state " << stateA << " and parser terminal " << eventLambda << " (" << rhs.NextTerminal().Str() << "), but could not be found." << std::endl;
        throw Exception("Gp2Pp", errstr.str(), 1001);
      }
      Idx aLambda = statemapit->second;
      
      //get state (B, lambda)
      statemapit = stateMap.find(std::make_pair(stateB,eventLambda));
      if(statemapit == stateMap.end()){
        std::stringstream errstr;
        errstr << "While executing Gp2Pp(): Tried to find a pushdown generator state for the pair of goto generator state " << stateB << " and parser terminal " << eventLambda << " (" << rhs.NextTerminal().Str() << "), but could not be found." << std::endl;
        throw Exception("Gp2Pp", errstr.str(), 1001);
      }
      Idx bLambda = statemapit->second;
      
      //get state (A, a)
      statemapit = stateMap.find(std::make_pair(stateA,eventA));
      if(statemapit == stateMap.end()){
        std::stringstream errstr;
        errstr << "While executing Gp2Pp(): Tried to find a pushdown generator state for the pair of goto generator state " << stateA << " and parser terminal " << eventA << " (" << lhs.NextTerminal().Str() << "), but could not be found." << std::endl;
        throw Exception("Gp2Pp", errstr.str(), 1001);
      }
      Idx aA = statemapit->second;
      
      //create stack symbol vectors lambda and A
      std::vector<StackSymbol> popLambda, pushA;
      popLambda.push_back(StackSymbol(FAUDES_PD_LAMBDA));
      std::stringstream s;
      s << stateA;
      pushA.push_back(StackSymbol(s.str()));
      
      //insert transition ((A, lambda), a, lambda, A, (B, lambda))
      rPd.SetTransition(aLambda, eventA, bLambda, popLambda, pushA);
      
      //insert transition ((A, a), lambda, lambda, A, (B, lambda))
      rPd.SetTransition(aA, eventLambda, bLambda, popLambda, pushA);
      
      
    }
    
    //for each reduce action (AwB | y) -> (AC | y)
    else if(//lhs.StateStack().size() >= 2 &&
      rhs.StateStack().size() == 2 && 
      lhs.StateStack().front() == rhs.StateStack().front() &&
      lhs.NextTerminal() == rhs.NextTerminal())
    {
      
      //extract A, B, C, w and y
      Idx stateA = rhs.StateStack().front();
      Idx stateB = lhs.StateStack().back();
      Idx stateC = rhs.StateStack().back();
      Idx eventY = lhs.NextTerminal().Event();
      std::vector<Idx> statesW;
      if(lhs.StateStack().size() > 2){
        statesW = std::vector<Idx>(lhs.StateStack().begin() + 1, lhs.StateStack().end() - 1);
      }
      
      //get state (B, lambda)
      Idx eventLambda = rPd.EventIndex(FAUDES_PD_LAMBDA);
      statemapit = stateMap.find(std::make_pair(stateB,eventLambda));
      if(statemapit == stateMap.end()){
        std::stringstream errstr;
        errstr << "While executing Gp2Pp(): Tried to find a pushdown generator state for the pair of goto generator state " << stateB << " and parser terminal " << eventLambda << " (" << FAUDES_PD_LAMBDA << "), but could not be found." << std::endl;
        throw Exception("Gp2Pp", errstr.str(), 1001);
      }
      Idx bLambda = statemapit->second;
      
      //get state (C, y)
      statemapit = stateMap.find(std::make_pair(stateC,eventY));
      if(statemapit == stateMap.end()){
        std::stringstream errstr;
        errstr << "While executing Gp2Pp(): Tried to find a pushdown generator state for the pair of goto generator state " << stateC << " and parser terminal " << eventY << " (" << lhs.NextTerminal().Str() << "), but could not be found." << std::endl;
        throw Exception("Gp2Pp", errstr.str(), 1001);
      }
      Idx cY = statemapit->second;
      
      //get state (B, y)
      statemapit = stateMap.find(std::make_pair(stateB,eventY));
      if(statemapit == stateMap.end()){
        std::stringstream errstr;
        errstr << "While executing Gp2Pp(): Tried to find a pushdown generator state for the pair of goto generator state " << stateB << " and parser terminal " << eventY << " (" << lhs.NextTerminal().Str() << "), but could not be found." << std::endl;
        throw Exception("Gp2Pp", errstr.str(), 1001);
      }
      Idx bY = statemapit->second;
      
      //create stack symbol vectors w^-1 A and A
      std::vector<StackSymbol> popWA, pushA;
      std::vector<Idx>::const_reverse_iterator idxit;
      for(idxit = statesW.rbegin(); idxit != statesW.rend(); idxit++){
        std::stringstream s;
        s << *idxit;
        popWA.push_back(StackSymbol(s.str()));
      }
      std::stringstream s;
      s << stateA;
      //only pop A if AwB has a size greater than 1
      if(lhs.StateStack().size() > 1){
        popWA.push_back(StackSymbol(s.str()));
      }
      else{
        popWA.push_back(StackSymbol(FAUDES_PD_LAMBDA));
      }
      pushA.push_back(StackSymbol(s.str()));
      
      //insert transition ((B, lambda), eventY, w^-1 A, A, (C, y))
      rPd.SetTransition(bLambda, eventY, cY, popWA, pushA);
      
      //insert transition ((B, y), lambda, w^-1 A, A, (C, y))
      rPd.SetTransition(bY, eventLambda, cY, popWA, pushA);
      
    }
    
    //every other action must be an error
    else{
      std::stringstream errstr;
      errstr << "While executing Gp2Pp(): Tried to create transition from parser action " << actit->Str() << ", but no rule exists for converting it. It seems to be neither a shift nor a reduce action. " << std::endl;
      throw Exception("Aug", errstr.str(), 1001);
    }
  }
  
  
  return rPd;
}

/* *************************
 * Dim
 * *************************/
PushdownGenerator Dim(const PushdownGenerator& pd, const Terminal& augSymbol){
  
  //generator to return
  PushdownGenerator rPd = pd;
  
  TransSet::Iterator transit;
  PopPushSet::const_iterator ppit;
  //look at every transition
  for(transit = pd.TransRelBegin(); transit != pd.TransRelEnd(); transit++){
    
    //if the event is the same as augSymbol
    if(transit->Ev == augSymbol.Event()){
      
      //set that transition's start state as final
      rPd.InsMarkedState(transit->X1);
      
      //delete that transition from the generator for every pop push set
      for(ppit = pd.PopPushBegin(*transit); ppit != pd.PopPushEnd(*transit); ppit++){
        rPd.ClrTransition(*transit, ppit->first, ppit->second);
      }
    }
  }
  
  //remove the event
  rPd.DelEvent(augSymbol.Event());
  
  return rPd;
}

} // namespace faudes

