/** @file pd_alg_nb_sub_b.h  Nonblock subfunctions, part B*/


/* Pushdown plugin for FAU Discrete Event Systems Library (libfaudes)

   Copyright (C) 2013  Stefan Jacobi, Sven Schneider, Anne-Kathrin Hess

*/


#ifndef FAUDES_PD_ALG_NB_SUB_B_H
#define FAUDES_PD_ALG_NB_SUB_B_H

#include "corefaudes.h"
#include "pd_pdgenerator.h"


namespace faudes {
  
  /**
   * Associates each state of the generator with an annotation. Each state's
   * merge attribute will be set to a MergeStateAnnotation. The MergeStateAnnotation 
   * will contain the annotation.
   * 
   * @param word
   *    word with which the states will be annotated
   * @param pd
   *    pushdown automaton to be annotated
   * @return
   *    copy of the parameter automaton with annotated states
   */
  PushdownGenerator RenQ(const std::string word, const PushdownGenerator& pd);
  
  /**
   * Adds an annotation to each stack symbol of the generator. Each stack symbol will
   * be copied and added to to generator with the annotation preceding the original
   * name. 
   * 
   * @param word
   *    word with which the stack symbols will be annotated
   * @param pd
   *    pushdown automaton to be annotated
   * @return
   *    copy of the parameter automaton with annotated states
   */  
  PushdownGenerator RenG(const std::string word, const PushdownGenerator& pd);
  
  /**
   * Removes all edges popping lambda.
   * 
   * @param pd
   *    pushdown automaton
   * @return
   *    pushdown automaton without lambda popping edges
   */
  PushdownGenerator Rep0(const PushdownGenerator& pd);
  
    /**
   * Restricts a pushdown generator to a pushdown generator with transitions that
   * are either read (p,a,lambda,lambda,q), pop (p,lambda,x,lambda,q) or 
   * push (p,lambda,x,yx,q)
   * 
   * @param pd
   *    the generator, which must not have transitions that pop more than one stack
   * symbol
   * @return 
   *    generator
   */
  PushdownGenerator Rpp(const PushdownGenerator& pd);
  
  /**
   * Removes transitions popping more than one stack symbol. Inserts new transitions
   * instead that only pop one stack symbol.
   * 
   * @param pd
   *    the generator, which must NOT contain lambda pops
   * @return 
   *    generator with altered transitions
   */
  PushdownGenerator Rep2(const PushdownGenerator& pd);
  
  /**
   * For debugging, prints a transitions with their pop and push attributes
   *
   * @param transitions
   *    multimap containing the transitions
   */
  void PrintTransitions(const std::multimap<Transition, std::pair<std::vector<StackSymbol>, std::vector<StackSymbol> > >& transitions);
  
  /**
   * Input must have passed Rpp!
   * remodells the generator to prevent double (or multiple) acceptance of the same
   * input string.
   * 
   * @param pd
   *    the generator
   * @return
   *    remodelled generator without double acceptance
   */
  PushdownGenerator Nda(const PushdownGenerator& pd);


} // namespace faudes

#endif
