/** @file con_decomposability.cpp Conditionaldecomposability */

/*
 * Implementation of the Conditionally decomposable algorithm
 *
 * Copyright (C) 2011  Tomas Masopust
 *
 */


#include "con_decomposability.h"
#include <vector>

namespace faudes {

bool IsConditionalDecomposable(const Generator& gen, const EventSetVector& rAlphabets, const EventSet& ek, Generator& proof) {
	FD_DF("Conditionaldecomposability checking...");
	
	// the generator must be deterministic
	if (gen.IsDeterministic() == false) {
	  std::stringstream errstr;
	  errstr << "Generator must be deterministic, but is nondeterministic";
	  throw Exception("ConditionalDecomposability", errstr.str(), 201);
	}

	// at least two alphabets in the vector
	if (rAlphabets.Size() < 2) {
	  std::stringstream errstr;
	  errstr << "At least two alphabets must be included in the EventSetVector";
	  throw Exception("ConditionalDecomposability", errstr.str(), 201);
	}

	Idx i;
    	EventSetVector ee = rAlphabets;		// Vector of input alphabets
	EventSet unionset;			// contains union of Ei
	EventSet shared;			// contains union of intersections

	// Compute unionset
	for (i = 0; i < ee.Size(); i++) {
		SetUnion(unionset,ee.At(i),unionset);
	}

	// Compute the set of shared events
	for (i = 0; i < ee.Size(); i++) {
	  for (Idx j = 0; j < ee.Size(); j++) {
	    if (j != i) {
		EventSet eHelpInt;
		SetIntersection(ee.At(i),ee.At(j),eHelpInt);
		SetUnion(shared,eHelpInt,shared);
	    }
	  }
	}

  	// Alphabet of the generator must be under union Ei 
	bool ok = SetInclusion(gen.Alphabet(),unionset);
  	if (ok == false) {
	  std::stringstream errstr;
	  errstr << "Generator alphabet is not included in union of the alphabets";
	  throw Exception("ConditionalDecomposability", errstr.str(), 100);
  	}

  	// Alphabet of the generator must contain Ek
	ok = SetInclusion(ek,gen.Alphabet());
  	if (ok == false) {
	  std::stringstream errstr;
	  errstr << "Generator alphabet does not include the alphabet ek";
	  throw Exception("ConditionalDecomposability", errstr.str(), 100);
  	}

  	// Ek must contain all shared events
	ok = SetInclusion(shared,ek);
  	if (ok == false) {
	  std::stringstream errstr;
	  errstr << "Ek does not include all shared events";
	  throw Exception("ConditionalDecomposability", errstr.str(), 100);
  	}

	// Make copies of the generator
	GeneratorVector copies;
	EventSet unionsetNew = unionset;		// unionset with new events
	EventSet::Iterator eit;

	for (i = 0; i < ee.Size(); i++) {
		EventSet uEiEk;
		SetUnion(ee.At(i),ek,uEiEk);
		EventSet EiEkdifUnionset;
		SetDifference(unionset,uEiEk,EiEkdifUnionset);
		Generator copy = gen;
		for (eit = EiEkdifUnionset.Begin(); eit != EiEkdifUnionset.End(); ++eit) {
			copy.EventRename(*eit,copy.UniqueEventName(""));
		}
		copies.Append(copy);
		SetUnion(unionsetNew,copy.Alphabet(),unionsetNew);
	}
	
	// Compute tilde G
	Generator tildeG = copies.At(0);
	for (i = 1; i < copies.Size(); i++) {
		Parallel(tildeG,copies.At(i),tildeG);
	}

	// Compute Inverse image of G
	Generator invGen;
	aInvProject(gen,unionsetNew,invGen);
	Trim(tildeG);
	Trim(invGen);
	if (LanguageInclusion(tildeG,invGen)) {
		Generator p;
		proof = p;	// the proof language is empty
		return true;
	}
	
	// Save proof automaton showing that it is not CD
	LanguageComplement(invGen);
	Generator proofGen;
	LanguageIntersection(invGen,tildeG,proofGen);
	Trim(proofGen);
	//StateMin(proofGen,proofGen);
	proofGen.ClearStateNames();
	proof = proofGen;
	
	return false;
}


} // name space 



