/** @file pd_alg_nb_sub_b.cpp  Nonblock subfunctions, part B */


/* Pushdown plugin for FAU Discrete Event Systems Library (libfaudes)

   Copyright (C) 2013  Stefan Jacobi, Sven Schneider, Anne-Kathrin Hess

*/

#include "pd_alg_nb_sub_b.h"

namespace faudes {

/* *************************
 * RenQ
 * *************************/
PushdownGenerator RenQ(const std::string word, const PushdownGenerator& pd){
    
  //take a copy of the old generator
  PushdownGenerator rPd = PushdownGenerator(pd);
  
  //annotate states of the new generator
  StateSet::Iterator its;    
  for(its = pd.StatesBegin(); its != pd.StatesEnd(); its++){
    MergeStateAnnotation msa = MergeStateAnnotation(*its,word);
    rPd.SetMerge(*its,msa);
  }
  return rPd;
}

/* *************************
 * RenG
 * *************************/
PushdownGenerator RenG(const std::string word, const PushdownGenerator& pd){
  
  //the annotation string
  std::string annotation = word + "-";
  
  //take a copy of the old generator
  PushdownGenerator rPd = PushdownGenerator(pd);
  
  //rename every stack symbol and delete the old one
  StackSymbolSet::Iterator it;
  for(it = pd.StackSymbolsBegin(); it != pd.StackSymbolsEnd(); it++){
    //dont rename lambda
    if(!pd.IsStackSymbolLambda(*it)){
      rPd.InsStackSymbol(annotation + pd.StackSymbolName(*it));
      rPd.DelStackSymbol(pd.StackSymbolName(*it));
    }
  }
  //rename stack bottom
  rPd.SetStackBottom(annotation + pd.StackSymbolName(pd.StackBottom()));
  
  //iterate over transitions and replace stack symbols in push and pop with 
  //the renamed versions
  TransSet::Iterator tit;
  std::vector<Idx> oldPush, oldPop, push, pop;
  std::vector<Idx>::const_iterator pushit, popit;
  PopPushSet popPush;
  PopPushSet::const_iterator ppit;
  
  //take copy of transitions to iterate over
  //copying is necessary because transitions will get deleted
  //iterating over the original TransSet can lead to undefined behaviour
  TransSet transRel = rPd.TransRel();
  for(tit = transRel.Begin(); tit != transRel.End(); tit++){
    
    //iterate over pop/push pairs
    popPush = rPd.PopPush(*tit);
    for(ppit = popPush.begin(); ppit != popPush.end(); ppit++){
      
      //rename pop stack symbols
      oldPop = ppit->first;
      for(popit = oldPop.begin(); popit != oldPop.end(); popit++){
        //dont rename lambda
        if(pd.IsStackSymbolLambda(*popit)){
          pop.push_back(*popit);
        }
        else{
          pop.push_back(rPd.StackSymbolIndex(annotation + rPd.StackSymbolName(*popit)));
        }
      }
      
      //rename push stack symbols
      oldPush = ppit->second;
      for(pushit = oldPush.begin(); pushit != oldPush.end(); pushit++){
        //dont rename lambda
        if(pd.IsStackSymbolLambda(*pushit)){
          push.push_back(*pushit);
        }
        else{
          push.push_back(rPd.StackSymbolIndex(annotation + rPd.StackSymbolName(*pushit)));
        }
      }
      
      //create new transitions
      rPd.SetTransition(*tit, pop, push);
      pop.clear();
      push.clear();
    
      //and delete the old one
      rPd.ClrTransition(*tit, oldPop, oldPush);
    }
  }
  return rPd;
}

/* *************************
 * Rep0
 * *************************/
PushdownGenerator Rep0(const PushdownGenerator& pd){
  
  //rename stack symbols and states
  PushdownGenerator rPd = RenG("old",RenQ("old",pd));
  
  //remember important attributes of the generator before proceeding
  StackSymbolSet oldStackSymbols = rPd.StackSymbols();
  Idx oldInitState = rPd.InitState();
  std::string oldStackBottom = rPd.StackSymbolName(rPd.StackBottom());
  
  //clear old init state and create new init state which will be annotated with "new"
  //and derived from pd.InitState() in a merge state annotation
  rPd.ClrInitState(rPd.InitState());
  Idx newInitState = rPd.InsState();
  rPd.SetInitState(newInitState);
  MergeStateAnnotation msa(pd.InitState(),"new");
  rPd.SetMerge(newInitState, msa);
  
  //set new stack bottom
  rPd.SetStackBottom("new-" + pd.StackSymbolName(pd.StackBottom()));
  
  //if the old initial state was marked, the new initial state is marked as well
  if(pd.ExistsMarkedState(pd.InitState())){
    rPd.InsMarkedState(rPd.InitState());
  }
  
  //add transition that adds the old stack bottom symbol to the stack
  //maybe lambda event needs to be inserted first
  Idx lambdaIdx = rPd.InsEvent(FAUDES_PD_LAMBDA);
  std::vector<StackSymbol> pop, push;
  std::string newStackBottom = rPd.StackSymbolName(rPd.StackBottom());
  //pop new stack bottom
  pop.push_back(StackSymbol(newStackBottom));
  //push new stack bottom and old stack bottom
  push.push_back(StackSymbol(oldStackBottom));
  push.push_back(StackSymbol(newStackBottom));
  //add transition
  rPd.SetTransition(newInitState, lambdaIdx, oldInitState, pop, push);
  
  //iterate over all transitions
  //take copy of transitions to iterate over
  //copying is necessary because transitions will get deleted
  //iterating over the original TransSet can lead to undefined behaviour
  TransSet transRel = rPd.TransRel();
  TransSet::Iterator tit;
  StackSymbolSet::Iterator ssit;
  std::vector<Idx> examinedPop, examinedPush, newPop, newPush;
  for(tit = transRel.Begin(); tit != transRel.End(); tit++){
    
    //look for lambda popping transitions (s1,ev,s2,lambda,w)
    //only need to look at the first pop/push pair, because only deterministic
    //generator are considered. the generator would be nondeterministic if there
    //was a transition in addition to a lambda popping one
    examinedPop = rPd.PopPush(*tit).begin()->first;
    examinedPush = rPd.PopPush(*tit).begin()->second;
    if(pd.IsStackSymbolLambda(examinedPop.front())){
      
      //remove the transition
      rPd.ClrTransition(*tit, examinedPop, examinedPush);
      
      //add new transitions for every possible stack symbol u minus the just
      //inserted bottom and lambda so that it looks like (s1,ev,s2,u,uw) for
      //for(ssit = oldStackSymbols.Begin(); ssit != oldStackSymbols.End(); ssit++){
      for(ssit = rPd.StackSymbolsBegin(); ssit != rPd.StackSymbolsEnd(); ssit++){
        if(!rPd.IsStackSymbolLambda(*ssit)){
          //fill pop with stack symbol
          newPop.clear();
          newPop.push_back(*ssit);
          //fill push with examinedPush plus stack symbol
          newPush.clear();
          newPush.insert(newPush.end(), examinedPush.begin(), examinedPush.end());
          newPush.push_back(*ssit);
          //add transition
          rPd.SetTransition(*tit, newPop, newPush);
        }
      }
    }    
  }
  return rPd;  
}


/* *************************
 * Rpp
 * *************************/
PushdownGenerator Rpp(const PushdownGenerator& pd){
  
  //take a copy of the old generator
  PushdownGenerator rPd = PushdownGenerator(pd);
  
  //annotate states of the new generator
  StateSet::Iterator stateit;    
  for(stateit = pd.StatesBegin(); stateit != pd.StatesEnd(); stateit++){
    MergeStateAnnotation msa = MergeStateAnnotation(*stateit, "old");
    rPd.SetMerge(*stateit,msa);
  }
  
  //clear transition relation
  rPd.ClearTransRel();
  
  std::multimap<Transition, std::pair<std::vector<Idx>, std::vector<Idx> > > readSet, multiPushSet, popPushSet, unchangedSet;
  std::multimap<Transition, std::pair<std::vector<Idx>, std::vector<Idx> > >::iterator multimapit;
  std::pair<Transition, std::pair<std::vector<Idx>, std::vector<Idx> > > transition;
  std::pair<std::vector<Idx>, std::vector<Idx> > popPushPair;
  std::vector<Idx> pop, push, pushEnd;
  TransSet::Iterator transit;
  PopPushSet::const_iterator ppit;
  //partition transitions of old generator into four sets
  //because states have only been copied (and annotated), the state indices remain
  //the same for both generators
  for(transit = pd.TransRelBegin(); transit != pd.TransRelEnd(); transit++){
    for(ppit = pd.PopPushBegin(*transit); ppit != pd.PopPushEnd(*transit); ppit++){
      
      //get pop and push pair
      pop = ppit->first;
      push = ppit->second;
      
      //create transition and decide further down into which set it will be put
      transition = std::make_pair(*transit, *ppit);
      
      
      //first set contains all transitions whose event is not lambda and pop does not
      //equal push
      //(i. e., are of the form (p,a,x,y,q) with x != y)
      if(!pd.IsEventLambda(transit->Ev) && pop != push){
        readSet.insert(transition);
      }
      
      //second and third set do not read anything (they have lambda events)
      else if(pd.IsEventLambda(transit->Ev)){
        
        //filter out all push transitions (p,lambda,x,yx,q), they will not need to be changed
        if(push.size() == 2 && push.back() == pop.front()){
          unchangedSet.insert(transition);
          continue;
        }
        //second set contains all transitions that push, but don't read lambda and
        //don't effectively pop
        //(i. e., are of the form (p,lambda,x,yax,q) with a not empty)
        else if(push.size() > 1){
          if(push.back() == pop.front()){
            multiPushSet.insert(transition);
          }
          
          //third set contains all transitions that pop and push
          //(i. e., are of the form (p,lambda,x,ya,q) with a not empty, x != a)
          else{
            popPushSet.insert(transition);
          }
        }
        //also put transitions without lambda push into third set
        else if (!pd.IsStackSymbolLambda(push.front())) {
          popPushSet.insert(transition);
        }
        
        //put transitions with lambda push into fourth set
        else{
          unchangedSet.insert(transition);
        }
      }
      //put all remaining transitions into fourth set as well
      else{
        unchangedSet.insert(transition);
      }
    }
  }
  
  
  //add lambda to events in case it has not already been added
  Idx lambdaIdx = rPd.InsEvent(FAUDES_PD_LAMBDA);
  
  Idx newState;
  //for every transition (p,a,x,y,q) in the first set
  for(multimapit = readSet.begin(); multimapit != readSet.end(); multimapit++){
    
    //add new intermediate state and annotate it with (p,a,x,x,p)
    newState = rPd.InsState();
    pop = multimapit->second.first;
    push = multimapit->second.first;
    MergeTransition mt(multimapit->first.X1, multimapit->first.Ev, multimapit->first.X1, pop,push);
    rPd.SetMerge(newState, mt);
    
    //add transition to and from intermediate state
    //(p,a,x,x,newstate)
    rPd.SetTransition(multimapit->first.X1, multimapit->first.Ev, newState, pop, push);
    //(newstate,lambda,x,y,q)
    rPd.SetTransition(newState, lambdaIdx, multimapit->first.X2, pop, multimapit->second.second);
  }
  
  //for every transition (p,lambda,x,yax,q) with a not empty in the second set
  for(multimapit = multiPushSet.begin(); multimapit != multiPushSet.end(); multimapit++){
    
    //add new intermediate state and annotate it with (p,lambda,x,ax,p)
    newState = rPd.InsState();
    pop = multimapit->second.first;
    push = std::vector<Idx>(multimapit->second.second.end() - multimapit->second.first.size() - 1, multimapit->second.second.end());
    MergeTransition mt(multimapit->first.X1, multimapit->first.Ev, multimapit->first.X1, pop,push);
    rPd.SetMerge(newState, mt);
    
    //add transition to and from intermediate state
    //(p,lambda,x,ax,newstate)
    rPd.SetTransition(multimapit->first.X1, multimapit->first.Ev, newState, pop, push);
    //(newstate,lambda,a,ya,q)
    pop.clear();
    pop.push_back(multimapit->second.second.at(multimapit->second.second.size() - multimapit->second.first.size() - 1));
    push = std::vector<Idx>(multimapit->second.second.begin(), multimapit->second.second.end() - multimapit->second.first.size());
    rPd.SetTransition(newState, lambdaIdx, multimapit->first.X2, pop, push);
  }
  
  //for every transition (p,lambda,x,ya,q) with a not empty and x != a in the third set
  for(multimapit = popPushSet.begin(); multimapit != popPushSet.end(); multimapit++){
    
    //add new intermediate state and annotate it with (p,lambda,x,lambda,p)
    newState = rPd.InsState();
    pop = multimapit->second.first;
    push.clear();
    push.push_back(pd.StackSymbolIndex(FAUDES_PD_LAMBDA));
    MergeTransition mt(multimapit->first.X1, multimapit->first.Ev, multimapit->first.X1, pop,push);
    rPd.SetMerge(newState, mt);
    
    //add transition to and from intermediate state
    //(p,lambda,x,lambda,newstate)
    rPd.SetTransition(multimapit->first.X1, lambdaIdx, newState, pop, push);
    //(newstate,lambda,lambda,ya,q)
    pop.clear();
    pop.push_back(pd.StackSymbolIndex(FAUDES_PD_LAMBDA));
    push = multimapit->second.second;
    rPd.SetTransition(newState, lambdaIdx, multimapit->first.X2, pop, push);
  }
  
  //for every transitions in the fourth set
  for(multimapit = unchangedSet.begin(); multimapit != unchangedSet.end(); multimapit++){
    
    //read pop and push for convenience
    pop = multimapit->second.first;
    push= multimapit->second.second;
    
    //if it is a read only transition in the form of (p,a,x,x,q), changed
    //it to (p,a,lambda,lambda,q)
//     if(!rPd.IsEventLambda(multimapit->first.Ev) && !pop.front().IsLambda() && pop == push ){
//       pop.clear();
//       pop.push_back(StackSymbol(FAUDES_PD_LAMBDA));
//       push.clear();
//       push.push_back(StackSymbol(FAUDES_PD_LAMBDA));
//     }
//     
    //insert the transition again
    rPd.SetTransition(multimapit->first.X1, multimapit->first.Ev, multimapit->first.X2, pop, push);
  }
  
  //if changes were made, repeat recursively
  //if lambda popping edges were added, they need to be removed first
  if(popPushSet.size() != 0){
    rPd = Rpp(Rep0(rPd));
  }
  else if(readSet.size() + multiPushSet.size() != 0){
    rPd = Rpp(rPd);
  }
  
  return rPd;
}

/* *************************
 * PrintTransitions
 * *************************/
void PrintTransitions(const std::multimap<Transition, std::pair<std::vector<StackSymbol>, std::vector<StackSymbol> > >& transitions){
  std::cout << "\nPrint Transitions:\n" << std::endl;
  std::multimap<Transition, std::pair<std::vector<StackSymbol>, std::vector<StackSymbol> > >::const_iterator transit;
  std::vector<StackSymbol>::const_iterator ssit;
  std::stringstream s;
  
  for(transit = transitions.begin(); transit != transitions.end(); transit++){
     s << transit->first.X1 << " " << PushdownGenerator::GlobalEventSymbolTablep()->Symbol(transit->first.Ev) << " " << transit->first.X2 << " [";
    for(ssit = transit->second.first.begin(); ssit != transit->second.first.end(); ssit++){
      s << ssit->Symbol() << " ";
    }
    s << "] [";
    for(ssit = transit->second.second.begin(); ssit != transit->second.second.end(); ssit++){
      s << ssit->Symbol() << " ";
    }
    s << "]" << std::endl;
    
   
  }
  std::cout << s.str() << std::endl;
}

/* *************************
 * Rep2
 * *************************/
PushdownGenerator Rep2(const PushdownGenerator& pd){
 
  //take a copy of the old generator
  PushdownGenerator rPd = PushdownGenerator(pd);
  
  //add lambda to events in case it has not already been added
  Idx lambdaIdx = rPd.InsEvent(FAUDES_PD_LAMBDA);
  
  //annotate states of the new generator. the state indices are the same for both 
  //generators, only the annotations will be different
  StateSet::Iterator stateit;    
  for(stateit = pd.StatesBegin(); stateit != pd.StatesEnd(); stateit++){
    MergeStateAnnotation msa = MergeStateAnnotation(*stateit,"old");
    rPd.SetMerge(*stateit,msa);
  }
  
  std::multimap<Transition, std::pair<std::vector<Idx>, std::vector<Idx> > > offendingTransSet;
  std::pair<Transition, std::pair<std::vector<Idx>, std::vector<Idx> > > transition;
  std::vector<Idx> pop, push;
  std::set< std::pair<Idx, Idx> > intermediateStates;
  std::pair<Idx, Idx> intermediateStatePair;
  std::map<std::pair<Idx, Idx>, Idx> mapIntermediateStates;
  TransSet::Iterator transit;
  PopPushSet::const_iterator ppit;
  Idx newState;
  //prepare an empty push vector
  push.push_back(pd.StackSymbolIndex(FAUDES_PD_LAMBDA));
  
  //look for transitions that pop more than one stack symbol,
  //(i. e. (q,w,av,v',q') with |a| = 1, |v| > 0)), save the transitions
  //in offendingTransSet, and add intermediate states that will be connected later
  for(transit = pd.TransRelBegin(); transit != pd.TransRelEnd(); transit++){
    for(ppit = pd.PopPushBegin(*transit); ppit != pd.PopPushEnd(*transit); ppit++){
      if(ppit->first.size() > 1){
        
        //save transition
        transition = std::make_pair(*transit, *ppit);
        offendingTransSet.insert(transition);
        
        //add intermediate state annotated with ((old,q),lambda,a,lambda,(old,q)), but
        //only if it has not already been added for this particular pair of (q,a)
        intermediateStatePair = std::make_pair(transit->X1,ppit->first.front());
        if(intermediateStates.insert(intermediateStatePair).second){
          
          //add intermediate state
          newState = rPd.InsState();
          //and save it for connecting with transitions later on
          mapIntermediateStates.insert(std::make_pair(intermediateStatePair,newState));
          
          //create annotation
          pop.clear();
          pop.push_back(ppit->first.front());
          MergeTransition mt(transit->X1, lambdaIdx, transit->X1, pop, push);//note: transit->X1 from pd refers to (old,transit->X1) in the context of
          //rPd, because state indices in pd and rPd are the same (they are copies) with
          //only the annotations being different
          rPd.SetMerge(newState, mt);
        }
      }
    }
  }
  
  std::multimap<Transition, std::pair<std::vector<Idx>, std::vector<Idx> > >::iterator offendingit;
  Idx startState, intermediateState;
  Idx popSymbol;
  TransSet transCopy;
  PopPushSet ppCopy;
  //iterate over all offending transitions (q,w,av,v',q') to fix them
  for(offendingit = offendingTransSet.begin(); offendingit != offendingTransSet.end(); offendingit++){
    
    //take a copy of the generator's transitions for iterating, because transitions will
    //be deleted and this would mess with the iterator
    transCopy = rPd.TransRel();
    
    //iterate over all transitions (q,x,ay,y',q'') starting at the same state q
    //and popping the same stack symbol a
    startState = offendingit->first.X1;
    popSymbol = offendingit->second.first.front();
    //ensure q
    for(transit = transCopy.Begin(startState); transit != transCopy.End(startState); transit++){
      
      //take a copy of the transitions's pop/push pairs for iterating, because pairs
      //will be deleted and this would mess with the iterator
      ppCopy = rPd.PopPush(*transit);
      
      for(ppit = ppCopy.begin(); ppit != ppCopy.end(); ppit++){
        //ensure a and |ay| > 1 (i. e. |y| != 0)
        if(ppit->first.front() == popSymbol && ppit->first.size() > 1){
          
          //delete the transition
          rPd.ClrTransition(*transit, ppit->first, ppit->second);
          
          //and insert transition to and from corresponding intermediate state
          //get intermediate state index
          intermediateState = mapIntermediateStates.find(std::make_pair(startState,popSymbol))->second;
          
          //set transition to intermediate state
          pop.clear();
          pop.push_back(popSymbol);
          push.clear();
          push.push_back(pd.StackSymbolIndex(FAUDES_PD_LAMBDA));
          rPd.SetTransition(startState, lambdaIdx, intermediateState, pop, push);
          
          //set transition from intermediate state
          pop = ppit->first;
          pop.erase(pop.begin());
          push  = ppit->second;
          rPd.SetTransition(intermediateState, transit->Ev, transit->X2, pop, push);
          
        }
      }
    }
  }
  
  //if offending transitions were found, check again if there are offending
  //transitions left
  if(offendingTransSet.size() != 0){
    rPd = Rep2(rPd);
  }
  return rPd;
}


/* *************************
 * Nda
 * *************************/
PushdownGenerator Nda(const PushdownGenerator& pd){
 
  PushdownGenerator rPd = pd;
  
  //states and transition relation will be rebuilt, so delete the existing ones
  rPd.ClearTransRel();
  rPd.ClearStates();
  
  StateSet::Iterator stateit;
  Idx newStateAct, newStatePas;
  std::string active = "active";
  std::string passive = "passive";
  std::map<Idx,std::pair<Idx,Idx> > stateMap;
  //for insert an active and a passive state for each state of the old generator
  for(stateit = pd.StatesBegin(); stateit != pd.StatesEnd(); stateit++){
    
    //active state
    newStateAct = rPd.InsState();
    MergeStateAnnotation mssAct(*stateit,active);
    rPd.SetMerge(newStateAct,mssAct);
    
    //if the old state was a final state, the active state is final as well
    if(pd.ExistsMarkedState(*stateit)){
      rPd.SetMarkedState(newStateAct);
    }
    
    //if the old state was the starting state, the active state is a starting state
    //as well
    if(pd.ExistsInitState(*stateit)){
      rPd.SetInitState(newStateAct);
    }
    
    //passive state
    newStatePas = rPd.InsState();
    MergeStateAnnotation mssPas(*stateit,passive);
    rPd.SetMerge(newStatePas,mssPas);
    
    //save relation between old state and new states for later reference
    stateMap.insert(std::make_pair(*stateit,std::make_pair(newStateAct,newStatePas)));
  }
  
  TransSet::Iterator transit;
  PopPushSet::const_iterator ppit;
  std::vector<Idx> pop, push;
  Idx startState, endState, oldStartState, oldEndState, oldEvent;
  //insert transitions
  for(transit = pd.TransRelBegin(); transit != pd.TransRelEnd(); transit++){
    for(ppit = pd.PopPushBegin(*transit); ppit != pd.PopPushEnd(*transit); ppit++){
      
      //for convenience
      pop = ppit->first;
      push = ppit->second;
      oldStartState = transit->X1;
      oldEndState = transit->X2;
      oldEvent = transit->Ev;
      
      //for any reading transition
      if(!pd.IsEventLambda(oldEvent)){
        
        //connect active state to active state
        startState = stateMap.find(oldStartState)->second.first;
        endState = stateMap.find(oldEndState)->second.first;
        rPd.SetTransition(startState,oldEvent,endState,pop,push);
        
        //connect passive state to active state
        startState = stateMap.find(oldStartState)->second.second;
        rPd.SetTransition(startState,oldEvent,endState,pop,push);
      }
      
      
      else{
        
        //for any pushing and popping transition
        if(push.size() == 2 || pd.IsStackSymbolLambda(push.front())){
          
          //connect passive state to passive state
          startState = stateMap.find(oldStartState)->second.second;
          endState = stateMap.find(oldEndState)->second.second;
          rPd.SetTransition(startState,oldEvent,endState,pop,push);
          
          //if the old start state was marked, connect active state to passive state
          if(pd.ExistsMarkedState(oldStartState)){
            startState = stateMap.find(oldStartState)->second.first;
            rPd.SetTransition(startState,oldEvent,endState,pop,push);
          }
          
          //else connect active state to active state
          else{
            startState = stateMap.find(oldStartState)->second.first;
            endState = stateMap.find(oldEndState)->second.first;
            rPd.SetTransition(startState,oldEvent,endState,pop,push);
          }
        }
      
        //should never get here
        else{
          std::stringstream errstr;
          errstr << "While executing Nda(): Found transition which is neither read nor pop nor push, which is not allowed" << std::endl;
          throw Exception("Nda", errstr.str(), 1001);
        }
      }
    }
  }
  
  return rPd;  
}


} // namespace faudes

