/** @file pd_alg_nb_sub_a.cpp Nonblock subfunctions, part A */


/* Pushdown plugin for FAU Discrete Event Systems Library (libfaudes)

   Copyright (C) 2013  Stefan Jacobi, Sven Schneider, Anne-Kathrin Hess

*/

#include "pd_alg_nb_sub_a.h"

namespace faudes {

  
/* *************************
 * Filter
 * *************************/
std::set<Nonterminal> Filter(const std::set<Nonterminal>& symbolSet, const GrammarSymbolVector& w){
  
  //resulting set
  std::set<Nonterminal> rSet;
  
  GrammarSymbolVector::const_iterator wit;
  std::set<Nonterminal>::const_iterator ntit;
  //iterate over all symbols of the word
  for(wit = w.begin(); wit != w.end(); wit++){
    //test if current symbol is a nonterminal
    ConstNonterminalPtr nt = std::tr1::dynamic_pointer_cast<const Nonterminal>(*wit);
    if(nt != NULL){
          
      //look for the symbol in the nonterminal set
      ntit = symbolSet.find(*nt);
      
      //if the symbol was found, insert it into resulting set
      if(ntit != symbolSet.end()){
        rSet.insert(*ntit);
      }
    }
  }
  return rSet;
}


/* *************************
 * Rnpp1
 * *************************/
std::set<Nonterminal> Rnpp1(const Grammar& gr, const std::set<Nonterminal>& ntSet){
  

  std::set<Nonterminal> rSet, filterSet;
  //copy resulting set from set of already eliminated nonterminals
  rSet = ntSet;
  
  std::set<GrammarProduction>::const_iterator pit;
  //iterate over all productions

  for(pit = gr.GrammarProductions().begin(); pit != gr.GrammarProductions().end(); pit++){
    
    filterSet = Filter(gr.Nonterminals(),pit->Rhs());
    
    //test if the nonterminals on the right hand side of the production are all in
    //the set of eliminable nonterminals (i. e., they are eliminable)
    //if so, then the nonterminal on the left hand side is eliminable as well
    if(std::includes(ntSet.begin(), ntSet.end(), filterSet.begin(), filterSet.end())){
      //insert left hand side into set of eliminable terminals
      rSet.insert(pit->Lhs());
    }
  }
  return rSet;
}

/* *************************
 * Rnppl
 * *************************/
std::set<Nonterminal> Rnppl(const Grammar& gr, const std::set<Nonterminal>& ntSet){
  
  //get nonterminals that are eliminable in one step
  std::set<Nonterminal> rSet = Rnpp1(gr,ntSet);
  //look if that changed anything
  if(ntSet != rSet){
    //if any changes were detected, go on
    rSet = Rnppl(gr,rSet);
  }
  return rSet;
}

/* *************************
 * Rnpp
 * *************************/
Grammar Rnpp(const Grammar& gr){
  
  //the grammar to be returned. keep the start symbol
  Grammar rGr(gr.StartSymbol());
  
  //compute removable nonterminals
  std::set<Nonterminal> removableNts = Rnppl(gr,std::set<Nonterminal>());
  
  std::set<GrammarProduction> productions;
  //if the start symbol is a removable nonterminal

  if(removableNts.find(gr.StartSymbol()) != removableNts.end()){
    
    std::set<GrammarProduction>::const_iterator gpit;
    //iterate over grammar productions A -> w
    for(gpit = gr.GrammarProductionsBegin(); gpit != gr.GrammarProductionsEnd(); gpit++){
      
      //convert A to GrammarSymbolPtr and put it togther with w into a vector
      Nonterminal* a = new Nonterminal(gpit->Lhs());
      NonterminalPtr aPtr(a);
      GrammarSymbolVector symbols(gpit->Rhs());
      symbols.push_back(aPtr);
      
      std::set<Nonterminal> filteredSet;
      //filter all nonterminals that are in A and w
      filteredSet = Filter(gr.Nonterminals(), symbols);
      
      //test if filteredSet is a subset of the removable nonterminals
      if(std::includes(removableNts.begin(), removableNts.end(), filteredSet.begin(), filteredSet.end())){
        productions.insert(*gpit);
      }
    }
  }
  
  //keep the terminals
  rGr.InsTerminals(gr.Terminals());
  
  //keep only removable nonterminals
  rGr.InsNonterminals(removableNts);
  
  //keep only the productions from above
  rGr.InsGrammarProductions(productions);
  
  return rGr;  
}

/* *************************
 * Sp2Lr
 * *************************/
Grammar Sp2Lr(const PushdownGenerator& pd){
  
  Grammar rGr;
    
  StateSet::Iterator stateit1, stateit2;
  EventSet::Iterator eventit;
  TransSet::Iterator transit;
  StackSymbolSet::Iterator ssit;
  PopPushSet::const_iterator ppit;
  std::vector<Idx> ssVector, pop, push;
  std::vector<Idx>::const_iterator popit, pushit;
  GrammarSymbolVector rhs;
  
  //Terminals
  
  //terminals equal the generator's events
  for(eventit = pd.AlphabetBegin(); eventit != pd.AlphabetEnd(); eventit++){
    rGr.InsTerminal(Terminal(*eventit));
  }
  
  //Nonterminals

  //for every state
  for(stateit1 = pd.StatesBegin(); stateit1 != pd.StatesEnd(); stateit1++){
    
    //insert end nonterminal for every stack symbol
    for(ssit = pd.StackSymbolsBegin(); ssit != pd.StackSymbolsEnd(); ssit++){
      if(!pd.IsStackSymbolLambda(*ssit)){
        ssVector.clear();
        ssVector.push_back(*ssit);
        rGr.InsNonterminal(Nonterminal(*stateit1,ssVector));
      }
    }
    
    //insert mid nonterminal for every state and every stack symbol
    for(stateit2 = pd.StatesBegin(); stateit2 != pd.StatesEnd(); stateit2++){
      for(ssit = pd.StackSymbolsBegin(); ssit != pd.StackSymbolsEnd(); ssit++){
        if(!pd.IsStackSymbolLambda(*ssit)){
          ssVector.clear();
          ssVector.push_back(*ssit);
          rGr.InsNonterminal(Nonterminal(*stateit1,ssVector,*stateit2));
        }
      }
    }
  }
  //Start Symbol
  ssVector.clear();
  ssVector.push_back(pd.StackBottom());
  rGr.SetStartSymbol(Nonterminal(pd.InitState(),ssVector));
  
  //Grammar Productions
  //for every transition
  for(transit = pd.TransRelBegin(); transit != pd.TransRelEnd(); transit++){
    for(ppit = pd.PopPushBegin(*transit); ppit != pd.PopPushEnd(*transit); ppit++){
      
      pop = ppit->first;
      std::vector<Idx> b = pop;
      push = ppit->second;
      Idx qi = transit->X1;
      Idx qj = transit->X2;
      Idx a = transit->Ev;
      
      //read transition (qi,a,b,b,qj)
      if(!pd.IsEventLambda(transit->Ev)){
        
        //insert production (qi,b) --> a(qj,b)
        Nonterminal ntLhs(qi,b);
        Nonterminal* ntRhs = new Nonterminal(qj,b);
        GrammarSymbolPtr ntRhsPtr(ntRhs);
        Terminal* t = new Terminal(a);
        GrammarSymbolPtr tPtr(t);
        rhs.clear();
        rhs.push_back(tPtr);
        rhs.push_back(ntRhsPtr);
        rGr.InsGrammarProduction(GrammarProduction(ntLhs,rhs));
        
        //for every state qt
        for(stateit1 = pd.StatesBegin(); stateit1 != pd.StatesEnd(); stateit1++){
          
          Idx qt = *stateit1;
          
          //insert production (qi,b,qt) --> a (qj,b,qt)
          Nonterminal ntLhs(qi,b,qt);
          Nonterminal* ntRhs = new Nonterminal(qj,b,qt);
          GrammarSymbolPtr ntRhsPtr(ntRhs);
          rhs.clear();
          rhs.push_back(tPtr);
          rhs.push_back(ntRhsPtr);
          rGr.InsGrammarProduction(GrammarProduction(ntLhs,rhs));
        }
      }
      else{
       
        //pop transition (qi,lambda,b,lambda,qj)
        if(pd.IsStackSymbolLambda(push.front())){
          
          //insert production (qi,b,qj) --> lambda
          Nonterminal ntLhs(qi,b,qj);
          Terminal* t = new Terminal(transit->Ev);
          GrammarSymbolPtr tPtr(t);
          rhs.clear();
          rhs.push_back(tPtr);
          rGr.InsGrammarProduction(GrammarProduction(ntLhs,rhs));
        }
        
        //push transition (qi,lambda,b,cb,qj)
        else if(push.size() == 2){
          
          std::vector<Idx> c;
          c.push_back(push.front());
          
          //insert production (qi,b) --> (qj,c)
          Nonterminal ntLhs(qi,b);
          Nonterminal* ntRhs = new Nonterminal(qj,c);
          GrammarSymbolPtr ntRhsPtr(ntRhs);
          rhs.clear();
          rhs.push_back(ntRhsPtr);
          rGr.InsGrammarProduction(GrammarProduction(ntLhs,rhs));
          
          //for every state qs
          for(stateit1 = pd.StatesBegin(); stateit1 != pd.StatesEnd(); stateit1++){
            
            Idx qs = *stateit1;
            
            //insert production (qi,b) --> (qj,c,qs)(qs,b)
            Nonterminal ntLhs(qi,b);
            Nonterminal* ntRhs1 = new Nonterminal(qj,c,qs);
            GrammarSymbolPtr ntRhs1Ptr(ntRhs1);
            Nonterminal* ntRhs2 = new Nonterminal(qs,b);
            GrammarSymbolPtr ntRhs2Ptr(ntRhs2);
            rhs.clear();
            rhs.push_back(ntRhs1Ptr);
            rhs.push_back(ntRhs2Ptr);
            rGr.InsGrammarProduction(GrammarProduction(ntLhs,rhs));
            
            //for every state qt
            for(stateit2 = pd.StatesBegin(); stateit2 != pd.StatesEnd(); stateit2++){
              
              Idx qt = *stateit2;
            
              //insert production (qi,b,qt) --> (qj,c,qs)(qs,b,qt)
              Nonterminal ntLhs(qi,b,qt);
              Nonterminal* ntRhs1 = new Nonterminal(qj,c,qs);
              GrammarSymbolPtr ntRhs1Ptr(ntRhs1);
              Nonterminal* ntRhs2 = new Nonterminal(qs,b,qt);
              GrammarSymbolPtr ntRhs2Ptr(ntRhs2);
              rhs.clear();
              rhs.push_back(ntRhs1Ptr);
              rhs.push_back(ntRhs2Ptr);
              rGr.InsGrammarProduction(GrammarProduction(ntLhs,rhs));
            }
          }
        }
      }
    }
  }

  //for every final state q and every stack symbol b
  for(stateit1 = pd.MarkedStatesBegin(); stateit1 != pd.MarkedStatesEnd(); stateit1++){
    for(ssit = pd.StackSymbolsBegin(); ssit != pd.StackSymbolsEnd(); ssit++){
      if(!pd.IsStackSymbolLambda(*ssit)){
        
        
        //insert a production ((q,b) --> lambda)
        ssVector.clear();
        ssVector.push_back(*ssit);
        Nonterminal nt(*stateit1,ssVector);
        Terminal* t = new Terminal(pd.EventIndex(FAUDES_PD_LAMBDA));
        GrammarSymbolPtr tPtr(t);
        GrammarSymbolVector v;
        v.push_back(tPtr);
        GrammarProduction gp(nt,v);
        rGr.InsGrammarProduction(gp);
      }
    }
  }
  return rGr;
}

/* *************************
 * Sp2Lr2
 * *************************/
Grammar Sp2Lr2(const PushdownGenerator& pd){
  
  Grammar rGr;
    
  StateSet::Iterator stateit1, stateit2;
  EventSet::Iterator eventit;
  TransSet::Iterator transit;
  StackSymbolSet::Iterator ssit;
  PopPushSet::const_iterator ppit;
  std::vector<Idx> ssVector, pop, push;
  std::vector<Idx>::const_iterator popit, pushit;
  GrammarSymbolVector rhs;
  
  //Terminals
  
  //terminals equal the generator's events
  for(eventit = pd.AlphabetBegin(); eventit != pd.AlphabetEnd(); eventit++){
    rGr.InsTerminal(Terminal(*eventit));
  }
  //Start Symbol
  ssVector.clear();
  ssVector.push_back(pd.StackBottom());
  rGr.SetStartSymbol(Nonterminal(pd.InitState(),ssVector));
  
  //Grammar Productions
  
  //for every final state q and every stack symbol b
  for(stateit1 = pd.MarkedStatesBegin(); stateit1 != pd.MarkedStatesEnd(); stateit1++){
    for(ssit = pd.StackSymbolsBegin(); ssit != pd.StackSymbolsEnd(); ssit++){
      if(!pd.IsStackSymbolLambda(*ssit)){
        
        //save used nonterminal
        ssVector.clear();
        ssVector.push_back(*ssit);
        Nonterminal nt(*stateit1,ssVector);
        rGr.InsNonterminal(nt);
        
        //insert a production ((q,b) --> lambda)
        Terminal* t = new Terminal(pd.EventIndex(FAUDES_PD_LAMBDA));
        GrammarSymbolPtr tPtr(t);
        GrammarSymbolVector v;
        v.push_back(tPtr);
        GrammarProduction gp(nt,v);
        rGr.InsGrammarProduction(gp);
        //std::cout << "inserting " << gp.Str() << std::endl;
      }
    }
  }
  
  
  //generate reducible productions until no more productions can be generated
  uint oldSize = 0;

  while(rGr.GrammarProductions().size() != oldSize){

    //save old number of grammar productions
    oldSize = rGr.GrammarProductions().size();
    
    //try to generate new grammar productions for each transition
    for(transit = pd.TransRelBegin(); transit != pd.TransRelEnd(); transit++){
      for(ppit = pd.PopPushBegin(*transit); ppit != pd.PopPushEnd(*transit); ppit++){
        
        pop = ppit->first;
        std::vector<Idx> b = pop;
        push = ppit->second;
        Idx qi = transit->X1;
        Idx qj = transit->X2;
        Idx a = transit->Ev;
        
        //read transition (qi,a,b,b,qj)
        if(!pd.IsEventLambda(transit->Ev)){
          
          //insert production (qi,b) --> a(qj,b)
          Nonterminal ntLhs(qi,b);
          Terminal* t = new Terminal(a);
          GrammarSymbolPtr tPtr(t);
          Nonterminal* ntRhs = new Nonterminal(qj,b);
          GrammarSymbolPtr ntRhsPtr(ntRhs);
          //... but only if (qj,b) is reducible
          if(rGr.Nonterminals().find(*ntRhs) != rGr.NonterminalsEnd()){ 
            
            //save lhs nonterminal as reducible
            rGr.InsNonterminal(ntLhs);
            
            //insert production
            rhs.clear();
            rhs.push_back(tPtr);
            rhs.push_back(ntRhsPtr);
            rGr.InsGrammarProduction(GrammarProduction(ntLhs,rhs));
            //std::cout << "inserting " << GrammarProduction(ntLhs,rhs).Str() << std::endl;
          }
          
          //for every state qt
          for(stateit1 = pd.StatesBegin(); stateit1 != pd.StatesEnd(); stateit1++){
            
            Idx qt = *stateit1;
            
            //insert production (qi,b,qt) --> a (qj,b,qt)
            Nonterminal ntLhs(qi,b,qt);
            Nonterminal* ntRhs = new Nonterminal(qj,b,qt);
            //... but only if (qj,b,qt) is reducible
            if(rGr.Nonterminals().find(*ntRhs) != rGr.NonterminalsEnd()){
              
              //save lhs nonterminal as reducible
              rGr.InsNonterminal(ntLhs);
              
              //insert production
              GrammarSymbolPtr ntRhsPtr(ntRhs);
              rhs.clear();
              rhs.push_back(tPtr);
              rhs.push_back(ntRhsPtr);
              rGr.InsGrammarProduction(GrammarProduction(ntLhs,rhs));
              //std::cout << "inserting " << GrammarProduction(ntLhs,rhs).Str() << std::endl;
            }
          }
        }
        else{
        
          //pop transition (qi,lambda,b,lambda,qj)
          if(pd.IsStackSymbolLambda(push.front())){
            
            //save lhs nonterminal as reducible
            Nonterminal ntLhs(qi,b,qj);
            rGr.InsNonterminal(ntLhs);
            
            //insert production (qi,b,qj) --> lambda
            Terminal* t = new Terminal(transit->Ev);
            GrammarSymbolPtr tPtr(t);
            rhs.clear();
            rhs.push_back(tPtr);
            rGr.InsGrammarProduction(GrammarProduction(ntLhs,rhs));
            //std::cout << "      inserting " << GrammarProduction(ntLhs,rhs).Str() << std::endl;
          }
          
          //push transition (qi,lambda,b,cb,qj)
          else if(push.size() == 2){
            
            std::vector<Idx> c;
            c.push_back(push.front());
            
            //insert production (qi,b) --> (qj,c)
            Nonterminal ntLhs(qi,b);
            Nonterminal* ntRhs = new Nonterminal(qj,c);
            GrammarSymbolPtr ntRhsPtr(ntRhs);
            //... but only if (qj,c) is reducible
            if(rGr.Nonterminals().find(*ntRhs) != rGr.NonterminalsEnd()){
              
              //save lhs nonterminal as reducible
              rGr.InsNonterminal(ntLhs);
              
              //insert production
              rhs.clear();
              rhs.push_back(ntRhsPtr);
              rGr.InsGrammarProduction(GrammarProduction(ntLhs,rhs));
              //std::cout << "inserting " << GrammarProduction(ntLhs,rhs).Str() << std::endl;
            }
            
            //for every state qs
            for(stateit1 = pd.StatesBegin(); stateit1 != pd.StatesEnd(); stateit1++){
              
              Idx qs = *stateit1;
              
              //insert production (qi,b) --> (qj,c,qs)(qs,b)
              Nonterminal ntLhs(qi,b);
              Nonterminal* ntRhs1 = new Nonterminal(qj,c,qs);
              GrammarSymbolPtr ntRhs1Ptr(ntRhs1);
              Nonterminal* ntRhs2 = new Nonterminal(qs,b);
              GrammarSymbolPtr ntRhs2Ptr(ntRhs2);
              //... but only if (qj,c,qs) and (qs,b) are reducible
              if(rGr.Nonterminals().find(*ntRhs1) != rGr.NonterminalsEnd() &&
                rGr.Nonterminals().find(*ntRhs2) != rGr.NonterminalsEnd()){
                
                //save lhs nonterminal as reducible
                rGr.InsNonterminal(ntLhs);
                
                //insert production
                rhs.clear();
                rhs.push_back(ntRhs1Ptr);
                rhs.push_back(ntRhs2Ptr);
                rGr.InsGrammarProduction(GrammarProduction(ntLhs,rhs));
                //std::cout << "inserting " << GrammarProduction(ntLhs,rhs).Str() << std::endl;
              }
              
              //for every state qt
              for(stateit2 = pd.StatesBegin(); stateit2 != pd.StatesEnd(); stateit2++){
                
                Idx qt = *stateit2;
              
                //insert production (qi,b,qt) --> (qj,c,qs)(qs,b,qt)
                Nonterminal ntLhs(qi,b,qt);
                Nonterminal* ntRhs1 = new Nonterminal(qj,c,qs);
                GrammarSymbolPtr ntRhs1Ptr(ntRhs1);
                Nonterminal* ntRhs2 = new Nonterminal(qs,b,qt);
                GrammarSymbolPtr ntRhs2Ptr(ntRhs2);
                //... but only if (qj,c,qs) and (qs,b,qt) are reducible
                if(rGr.Nonterminals().find(*ntRhs1) != rGr.NonterminalsEnd() &&
                  rGr.Nonterminals().find(*ntRhs2) != rGr.NonterminalsEnd()){
                  
                  //save lhs nonterminal as reducible
                  rGr.InsNonterminal(ntLhs);
                  
                  //insert production
                  rhs.clear();
                  rhs.push_back(ntRhs1Ptr);
                  rhs.push_back(ntRhs2Ptr);
                  rGr.InsGrammarProduction(GrammarProduction(ntLhs,rhs));
                  //std::cout << "inserting " << GrammarProduction(ntLhs,rhs).Str() << std::endl;
                }
              }
            }
          }
        }
      }
    }

  
  }
  return rGr;
}

/* *************************
 * Rup
 * *************************/
Grammar Rup(const Grammar& gr){
  Grammar rGr;
  
  //copy terminals and start symbol from the old grammar
  rGr.InsTerminals(gr.Terminals());
  rGr.SetStartSymbol(gr.StartSymbol());
  
  std::set<GrammarProduction> todoProductions, reachableProductions;
  std::set<GrammarProduction>::const_iterator gpit;
  std::set<Nonterminal> usedNonterminals;
  
  //start with the start symbol, which is always used in a grammar
  usedNonterminals.insert(gr.StartSymbol());
  
  //get productions with start symbol as righthand side
  for(gpit = gr.GrammarProductionsBegin(); gpit != gr.GrammarProductionsEnd(); gpit++){
    if(gpit->Lhs() == gr.StartSymbol()){
     todoProductions.insert(*gpit); 
    }
  }
  
  //look at all todo productions
  while(!todoProductions.empty()){
    
    //get the current todo production
    GrammarProduction currentGp(*todoProductions.begin());
    todoProductions.erase(currentGp);
    
    //since this production was reached, it is a reachable production
    reachableProductions.insert(currentGp);
    
    //insert all nonterminals of the current production into set of used nonterminals
    std::set<Nonterminal> filtered = Filter(gr.Nonterminals(),currentGp.Rhs());
    usedNonterminals.insert(filtered.begin(), filtered.end());
    
    //look at all productions
    for(gpit = gr.GrammarProductionsBegin(); gpit != gr.GrammarProductionsEnd(); gpit++){
      //if they have any of the used nonterminals on their lefthand side, insert them
      //into the todo productions
      if(usedNonterminals.find(gpit->Lhs()) != usedNonterminals.end()){
        todoProductions.insert(*gpit);
      }
    }
    
    //do not look at already reachable productions twice
    //this avoids infinite loops
    for(gpit = reachableProductions.begin(); gpit != reachableProductions.end(); gpit++){
      todoProductions.erase(*gpit);
    }
  }
  
  //insert all used nonterminals and all reachable productions
  rGr.InsNonterminals(usedNonterminals);
  rGr.InsGrammarProductions(reachableProductions);
  
  return rGr;
}
  

} // namespace faudes

