/** @file op_mc.cpp 

Method to verify mutual controllability for two given generators.
A definition of mutual controllability is given in 
S.-H. Lee and K. C. Wong, “Structural decentralised control of concurrent
DES,” European Journal of Control, vol. 35, pp. 1125-1134,2002.
*/

/* FAU Discrete Event Systems Library (libfaudes)

   Copyright (C) 2006  Bernd Opitz
   Exclusive copyright is granted to Klaus Schmidt

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, write to the Free Software
   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA */

#include "op_mc.h"

namespace faudes {

bool IsMutuallyControllable(const System& rGen1, const System& rGen2) {
	OP_DF("IsMutuallyControllable(" << rGen1.Name() << ", "
			<< rGen2.Name() << ")");
	// helpers:
	StateSet forbidden1;
	StateSet forbidden2;
	// Tha algorithm is implemented by the following function
	return IsMutuallyControllable(rGen1, rGen2, forbidden1, forbidden2);
}


bool IsMutuallyControllable(const System& rGen1, const System& rGen2,
		StateSet& rForbidden1, StateSet& rForbidden2) {
	OP_DF("IsMutuallyControllable(" << rGen1.Name() << ", "
			<< rGen2.Name() << ", rForbidden1, rForbidden2)");
	// algorithm:
	// Compute the intersection of the generator alphabets
	const EventSet alph12 = rGen1.Alphabet() * rGen2.Alphabet();
	OP_DF("IsMutuallyControllable: shared events \"" << alph12.ToString() 
			<< "\"");
	// check for shared events: if there are no shared events, mutual controllability is fulfilled anyway.
	if (alph12.Empty()) {
		OP_DF("IsMutuallyControllable: shared alphabet empty. "
				<< "terminating successful");
		return true;
	}
	// check for shared uncontrollable events
	const EventSet ualph12 = rGen1.UncontrollableEvents() * rGen2.UncontrollableEvents();
	OP_DF("IsMutuallyControllable: shared uncontrollable events \""
			<< ualph12.ToString() << "\"");
	// if there are no shared uncontrollable events, mutual controllability is fulfilled anyway.
	if (ualph12.Empty()) {
		OP_DF("IsMutuallyControllable: shared uncontrollable alphabet empty "
				<< "terminating successful");
		return true;
	}
	// prepare results
	bool result1;
	bool result2;
	rForbidden1.Clear();
	rForbidden2.Clear();
	// The generator g will serve as the plant in the subsequent controllability evaluation
	System g = rGen1.NewCGen();
	// "plant": (P_21)^-1( P_12( L(G_2) )) is the language generated by g
	aProject(rGen2, alph12, g);
	InvProject(g, rGen1.Alphabet());
	// The set of controllable events for the controllability test is alphabet of rGen1 - the shared uncontrollable events
	// mutual controllability test of rGen2 w.r.t. rGen1.
	result1 = IsControllable(g,  (rGen1.Alphabet() - ualph12), rGen1, rForbidden1);
	// print uncontrollable states if result is negative
	if (result1 == false) {
		OP_DF("IsMutuallyControllable(" << rGen1.Name() << ", " << rGen2.Name() 
				<< "): uncontrollable states: "	<< rForbidden1.ToString());
	}
	// "plant": (P_12)^-1( P_21( L(G_1) )) is the language generated by g
	aProject(rGen1, alph12, g);
	InvProject(g, rGen2.Alphabet());
	// The set of controllable events for the controllability test is alphabet of rGen2 - the shared uncontrollable events
	// mutual controllability test of rGen1 w.r.t. rGen2.
	result2 = IsControllable(g, (rGen2.Alphabet() - ualph12), rGen2, rForbidden2);
	// print uncontrollable states if result is negative
	if (result2 == false) {
		OP_DF("IsMutuallyControllable(" << rGen1.Name() << ", " << rGen2.Name() 
				<< "): uncontrollable states: "	<< rForbidden2.ToString());
	}
	
	if (result1 && result2) {
		OP_DF("IsMutuallyControllable: fulfilled");
	}
	else {
		OP_DF("IsMutuallyControllable: failed");
	}

	return (result1 && result2);
}

// RTI wrapper
void IsMutuallyControllable(const System& rGen1, const System& rGen2, bool& rRes) {
  rRes=IsMutuallyControllable(rGen1, rGen2);
}

} // namespace faudes

